from maps.wikimap.stat.assessment.graded_units_log.lib.commits import (
    get_edits_events,
    get_moderation_tasks,
    _get_task_id,
    add_task_id_region_id,
)
from maps.wikimap.stat.libs import nile_ut
from nile.api.v1 import Record
from qb2.api.v1 import typing
from yt.yson import to_yson_type
import pytest


def test_should_get_edits_events():
    result = nile_ut.yql_run(
        get_edits_events,
        commit_event_table=nile_ut.Table([
            Record(commit_id=1, created_at='2021-07-08suffix', bounds_geom='g1', primary_object_category_id='c1', type=to_yson_type('edit'),     test_column='t1'),
            Record(commit_id=2, created_at='2021-07-08suffix', bounds_geom='g2', primary_object_category_id='c2', type=to_yson_type('not-edit'), test_column='t2'),
        ]),
        min_action_date='2021-07-08',
    )
    assert sorted([
        Record(entity_id=1, geom='g1', category_id='c1'),
    ]) == sorted(result)


def test_should_filter_edit_events_by_min_action_date():
    result = nile_ut.yql_run(
        get_edits_events,
        commit_event_table=nile_ut.Table([
            Record(commit_id=1, created_at='2021-07-07suffix', bounds_geom='g1', primary_object_category_id='c1', type=to_yson_type('edit'), test_column='t1'),
            Record(commit_id=2, created_at='2021-07-08suffix', bounds_geom='g2', primary_object_category_id='c2', type=to_yson_type('edit'), test_column='t2'),
            Record(commit_id=3, created_at='2021-07-09suffix', bounds_geom='g3', primary_object_category_id='c3', type=to_yson_type('edit'), test_column='t3'),
        ]),
        min_action_date='2021-07-08',
    )
    assert sorted([
        Record(entity_id=2, geom='g2', category_id='c2'),
        Record(entity_id=3, geom='g3', category_id='c3'),
    ]) == sorted(result)


def test_should_get_moderation_tasks_filtered_by_min_action_date():
    result = nile_ut.yql_run(
        get_moderation_tasks,
        social_task_table=nile_ut.Table([
            # not resolved
            Record(event_id=1, primary_object_category_id='c1', resolved_at=None, closed_at=None, test_column1='t11'),

            # resolved, not closed
            Record(event_id=2, primary_object_category_id='c2', resolved_at='2021-07-07suffix', closed_at=None, test_column1='t12'),
            Record(event_id=3, primary_object_category_id='c3', resolved_at='2021-07-08suffix', closed_at=None, test_column1='t13'),
            Record(event_id=4, primary_object_category_id='c4', resolved_at='2021-07-09suffix', closed_at=None, test_column1='t14'),

            # closed
            Record(event_id=5, primary_object_category_id='c5', resolved_at='2021-07-07suffix', closed_at='2021-07-07suffix', test_column1='t15'),
            Record(event_id=6, primary_object_category_id='c6', resolved_at='2021-07-07suffix', closed_at='2021-07-08suffix', test_column1='t16'),
            Record(event_id=7, primary_object_category_id='c7', resolved_at='2021-07-08suffix', closed_at='2021-07-09suffix', test_column1='t17'),
        ]),
        commit_event_table=nile_ut.Table([
            Record(event_id=1, created_at='2021-07-07suffix', bounds_geom='g1', test_column2='t21'),

            Record(event_id=2, created_at='2021-07-07suffix', bounds_geom='g2', test_column2='t22'),
            Record(event_id=3, created_at='2021-07-08suffix', bounds_geom='g3', test_column2='t23'),
            Record(event_id=4, created_at='2021-07-09suffix', bounds_geom='g4', test_column2='t24'),

            Record(event_id=5, created_at='2021-07-07suffix', bounds_geom='g5', test_column2='t25'),
            Record(event_id=6, created_at='2021-07-07suffix', bounds_geom='g6', test_column2='t26'),
            Record(event_id=7, created_at='2021-07-07suffix', bounds_geom='g7', test_column2='t27'),
        ]),
        min_action_date='2021-07-08',
    )
    assert sorted([
        Record(entity_id=3, geom='g3', category_id='c3'),
        Record(entity_id=4, geom='g4', category_id='c4'),

        Record(entity_id=6, geom='g6', category_id='c6'),
        Record(entity_id=7, geom='g7', category_id='c7'),
    ]) == sorted(result)


def test_should_correctly_get_geometry_for_max_delayed_moderation_tasks():
    result = nile_ut.yql_run(
        get_moderation_tasks,
        social_task_table=nile_ut.Table(
            data=[
                Record(event_id=1, primary_object_category_id='c1', resolved_at='2021-07-23suffix', closed_at=None, test_column1='t11'),
                Record(event_id=2, primary_object_category_id='c2', resolved_at='2021-07-24suffix', closed_at=None, test_column1='t12'),
            ],
            schema={
                'event_id':                   typing.Int64,
                'primary_object_category_id': typing.Unicode,
                'resolved_at':                typing.Unicode,
                'closed_at':                  typing.Optional[typing.Unicode],
                'test_column1':               typing.Unicode,
            },
        ),
        commit_event_table=nile_ut.Table([
            Record(event_id=1, created_at='2021-07-07suffix', bounds_geom='g1', test_column2='t21'),
        ]),
        min_action_date='2021-07-08',
    )
    assert sorted([
        Record(entity_id=1, category_id='c1', geom='g1'),
    ]) == sorted(result)


def test_should_get_task_id():
    assert 'edits/action1/category1' == _get_task_id('edits', 'action1', 'category1')
    assert 'edits/action2/common' == _get_task_id('edits', 'action2', None)
    assert 'moderation/resolved/category1' == _get_task_id('moderation', 'resolve', 'category1')
    assert 'moderation/closed/common' == _get_task_id('moderation', 'close', None)


@pytest.fixture
def mock_geobase(monkeypatch):
    PATCHED_MODULE = 'maps.wikimap.stat.assessment.graded_units_log.lib.commits.'
    monkeypatch.setattr(PATCHED_MODULE + 'mercator_geom_to_region_id', int)
    monkeypatch.setattr(PATCHED_MODULE + 'GEOBASE_FILES', [])


def test_should_add_task_id_region_id(mock_geobase):
    result = nile_ut.yql_run(
        add_task_id_region_id,
        graded_units=nile_ut.Table([
            Record(entity_id='1', entity_domain='edits',      action='action1', test_column='t1'),
            Record(entity_id='2', entity_domain='edits',      action='action2', test_column='t2'),
            Record(entity_id='3', entity_domain='edits',      action='action3', test_column='t3'),
        ]),
        geom_and_category=nile_ut.Table([
            Record(entity_id=1, geom='101', category_id='category1'),
            Record(entity_id=2, geom='102', category_id='category2'),
            Record(entity_id=3, geom='103', category_id=None),
        ]),
    )
    assert sorted([
        Record(entity_id='1', entity_domain='edits', action='action1', region_id=101, task_id='edits/action1/category1', test_column='t1'),
        Record(entity_id='2', entity_domain='edits', action='action2', region_id=102, task_id='edits/action2/category2', test_column='t2'),
        Record(entity_id='3', entity_domain='edits', action='action3', region_id=103, task_id='edits/action3/common',    test_column='t3'),
    ]) == sorted(result)
