from maps.wikimap.stat.assessment.graded_units_log.lib.tracker import get_tracker_issues, add_task_id_region_id
from maps.wikimap.stat.libs import nile_ut
from maps.wikimap.stat.libs.geobase_regions import EARTH_REGION_ID
from nile.api.v1 import Record
from qb2.api.v1 import typing
from yt.yson import to_yson_type
import datetime

_ISSUE_EVENTS_SCHEMA = {
    'issue':   typing.Unicode,
    'date':    typing.Int64,
    'changes': typing.Yson,
}


def timestamp(date_iso_str):
    return int(datetime.datetime.fromisoformat(date_iso_str).timestamp()) * 1000


def test_should_get_tracker_issues():
    open_changes = to_yson_type([{b'field': b'status', b'newValue': {b'value': {b'id': b'status id open'}}}])
    closed_changes = to_yson_type([{b'field': b'status', b'newValue': {b'value': {b'id': b'status id closed'}}}])

    result = nile_ut.yql_run(
        get_tracker_issues,
        dates=['2020-01-25', '2020-01-26'],
        components_table=nile_ut.Table([
            Record(id='component id 1', name='Компонент 1'),
            Record(id='component id 2', name='Компонент 2'),
            Record(id='component id 3', name='Компонент 3'),
            Record(id='component id 4', name='Компонент 4'),
            Record(id='component id 5', name='Компонент 5'),
            Record(id='component id 6', name='Компонент 6'),
            Record(id='component id 7', name='Компонент 7'),  # Unused entry
        ]),
        issue_events_table=nile_ut.Table([
            # First closed at another day
            Record(issue='issue id 1', date=timestamp('2020-01-23'), changes=closed_changes),
            Record(issue='issue id 1', date=timestamp('2020-01-24'), changes=open_changes),
            Record(issue='issue id 1', date=timestamp('2020-01-25'), changes=closed_changes),

            # Non-closed
            Record(issue='issue id 2', date=timestamp('2020-01-26'), changes=open_changes),

            # Okay, but reopened later
            Record(issue='issue id 3', date=timestamp('2020-01-25'), changes=closed_changes),
            Record(issue='issue id 3', date=timestamp('2020-01-27'), changes=open_changes),

            # Okay
            Record(issue='issue id 4', date=timestamp('2020-01-26'), changes=closed_changes),

            # Unpaid resolution
            Record(issue='issue id 5', date=timestamp('2020-01-25'), changes=closed_changes),

            # Unpaid queue
            Record(issue='issue id 6', date=timestamp('2020-01-25'), changes=closed_changes)
        ], schema=_ISSUE_EVENTS_SCHEMA),
        issues_table=nile_ut.Table([
            Record(id='issue id 1', resolution='resolution id fixed', assignee=1, components=to_yson_type([b'component id 1']), key='GEOCONTENTFB-1', storyPoints=1),
            Record(id='issue id 2', resolution='resolution id fixed', assignee=2, components=to_yson_type([b'component id 2']), key='GEOCONTENTFB-2', storyPoints=2),
            Record(id='issue id 3', resolution='resolution id fixed', assignee=3, components=to_yson_type([b'component id 3']), key='GEOCONTENTFB-3', storyPoints=3),
            Record(id='issue id 4', resolution='resolution id fixed', assignee=4, components=to_yson_type([b'component id 4']), key='GEOCONTENTFB-4', storyPoints=4),
            Record(id='issue id 5', resolution='resolution id other', assignee=5, components=to_yson_type([b'component id 5']), key='GEOCONTENTFB-5', storyPoints=5),
            Record(id='issue id 6', resolution='resolution id fixed', assignee=5, components=to_yson_type([b'component id 6']), key='MAPSPW-6',       storyPoints=6),
        ]),
        resolutions_table=nile_ut.Table([
            Record(id='resolution id fixed',      key='fixed'),
            Record(id='resolution id other',      key='other'),
            Record(id='resolution id unused',     key='unused'),  # Unused entry
        ]),
        statuses_table=nile_ut.Table([
            Record(id='status id closed', key='closed'),
            Record(id='status id open',   key='open'),
            Record(id='status id unused', key='unused'),  # Unused entry
        ])
    )

    assert sorted([
        Record(entity_id='GEOCONTENTFB-3', component='Компонент 3'),
        Record(entity_id='GEOCONTENTFB-4', component='Компонент 4'),
    ]) == sorted(result)


def test_should_add_task_id_region_id():
    result = nile_ut.yql_run(
        add_task_id_region_id,
        tracker_units=nile_ut.Table([
            Record(entity_id='GEOCONTENTFB-1', test_column=91),
            Record(entity_id='GEOCONTENTFB-2', test_column=92),
        ]),
        tracker_issues=nile_ut.Table([
            Record(entity_id='GEOCONTENTFB-1', component='Компонент 1'),
            Record(entity_id='GEOCONTENTFB-2', component='Компонент 2'),
        ])
    )
    assert sorted([
        Record(entity_id='GEOCONTENTFB-1', task_id='tracker/geocontentfb/Компонент 1', region_id=EARTH_REGION_ID, test_column=91),
        Record(entity_id='GEOCONTENTFB-2', task_id='tracker/geocontentfb/Компонент 2', region_id=EARTH_REGION_ID, test_column=92),
    ]) == sorted(result)
