from maps.wikimap.stat.assessment.report.lib import util

from maps.wikimap.stat.libs.feedback import get_paid_operations_at_date, get_task_id

from nile.api.v1 import extractors


def _get_assessed_tasks(feedback_table, latest_grades):
    '''
    feedback_table:
    | id (uint) | position | source | type | ... |
    |-----------+----------+--------+------+-----|
    | ...       | ...      | ...    | ...  | ... |

    latest_grades:
    | entity_id (string) | action_by | action | graded_by | grade |
    |--------------------+-----------+--------+-----------+-------|
    | ...                | ...       | ...    | ...       | ...   |

    result:
    | position | source | type | action_by | action | graded_by | grade |
    |----------+--------+------+-----------+--------+-----------+-------|
    | ...      | ...    | ...  | ...       | ...    | ...       | ...   |
    '''
    latest_grades = latest_grades.project(
        extractors.all(),
        entity_id=util.extract_as_uint('entity_id')
    )

    return feedback_table.project(
        'position', 'source', 'type',
        entity_id='id'
    ).join(
        latest_grades,
        type='inner',
        by='entity_id'
    ).project(
        extractors.all(exclude='entity_id')
    )


def _get_paid_tasks_at_date(date, feedback_table, social_commit_event):
    '''
    feedback_table:
    | id  | position | source | type | history                  | commit_ids      | ... |
    |-----+----------+--------+------+--------------------------+-----------------+-----|
    | ... | ...      | ...    | ...  | [                        | [commitId, ...] | ... |
    |     |          |        |      |   {                      |                 |     |
    |     |          |        |      |     'operation':  bytes, |                 |     |
    |     |          |        |      |     'modifiedBy': int,   |                 |     |
    |     |          |        |      |     'modifiedAt': bytes, |                 |     |
    |     |          |        |      |     ...                  |                 |     |
    |     |          |        |      |   }                      |                 |     |
    |     |          |        |      | ]                        |                 |     |

    social_commit_event:
    | commit_id | created_at | type | ... |
    |-----------+------------+------+-----|
    | ...       | ...        | ...  | ... |

    result:
    | position | source | type          | action_by          | action            |
    |----------+--------+---------------+--------------------+-------------------|
    | ...      | ...    | feedback.type | history.modifiedBy | history.operation |
    '''
    paid_operations = get_paid_operations_at_date(date, feedback_table, social_commit_event)

    return paid_operations.project(
        extractors.all(exclude=['operation', 'modified_by', 'modified_at']),
        action_by='modified_by',
        action='operation'
    )


def _source_type_to_task_id(feedback_tasks):
    '''
    Converts `source`, `type` and `action` to `task_id`.

    feedback_tasks:
    | position | source | type | action_by | action | graded_by | grade |
    |----------+--------+------+-----------+--------+-----------+-------|
    | ...      | ...    | ...  | ...       | ...    | ...       | ...   |

    result:
    | position | task_id                                 | action_by | action | graded_by | grade |
    |----------+-----------------------------------------+-----------+--------+-----------+-------|
    | ...      | task_id_extractor(source, type, action) | ...       | ...    | ...       | ...   |
    '''
    return feedback_tasks.project(
        extractors.all(exclude=['source', 'type']),
        task_id=extractors.custom(
            get_task_id,
            'source',
            'type',
            'action',
        ),
    )


def _position_to_lat_lon(feedback_tasks_with_task_id):
    '''
    feedback_tasks_with_task_id:
    | position   | task_id | action_by | action | graded_by | grade |
    |------------+---------+-----------+--------+-----------+-------|
    | [lon, lat] | ...     | ...       | ...    | ...       | ...   |

    result:
    | lat          | lon          | task_id | action_by | action | graded_by | grade |
    |--------------+--------------+---------+-----------+--------+-----------+-------|
    | position.lat | position.lon | ...     | ...       | ...    | ...       | ...   |
    '''
    def position_to_lat_lon(records):
        for record in records:
            if ('position' not in record) or (record.position is None):
                yield record.transform('position', lat=None, lon=None)
                continue

            assert len(record.position) == 2
            lon, lat = record.position

            yield record.transform('position', lat=lat, lon=lon)

    return feedback_tasks_with_task_id.map(position_to_lat_lon)


def get_tasks(job, date, latest_grades, feedback_table, social_commit_event):
    '''
    latest_grades:
    | entity_id | action_by | action | graded_by | grade |
    |-----------+-----------+--------+-----------+-------|
    | ...       | ...       | ...    | ...       | ...   |

    feedback_table:
    | id  | position   | source | type | history                  | commit_ids      | ... |
    |-----+------------+--------+------+--------------------------+-----------------+-----|
    | ... | [lon, lat] | ...    | ...  | [                        | [commitId, ...] | ... |
    |     |            |        |      |   {                      |                 |     |
    |     |            |        |      |     'operation':  bytes, |                 |     |
    |     |            |        |      |     'modifiedBy': int,   |                 |     |
    |     |            |        |      |     'modifiedAt': bytes, |                 |     |
    |     |            |        |      |     ...                  |                 |     |
    |     |            |        |      |   }                      |                 |     |
    |     |            |        |      | ]                        |                 |     |

    social_commit_event:
    | commit_id | created_at | type | ... |
    |-----------+------------+------+-----|
    | ...       | ...        | ...  | ... |

    result:
    | lat          | lon          | task_id                                 | action_by | action | graded_by | grade |
    |--------------+--------------+-----------------------------------------+-----------+--------+-----------+-------|
    | position.lat | position.lon | task_id_extractor(source, type, action) | ...       | ...    | ...       | ...   |
    '''
    feedback_tasks = util.concat_assessed_and_paid_tasks(
        job,
        _get_assessed_tasks(feedback_table, latest_grades),
        _get_paid_tasks_at_date(date, feedback_table, social_commit_event)
    )
    feedback_tasks_with_task_id = _source_type_to_task_id(feedback_tasks)
    return _position_to_lat_lon(feedback_tasks_with_task_id)
