from maps.wikimap.stat.assessment.report.lib import util
from maps.wikimap.stat.libs.assessment import add_tracker_task_id

from nile.api.v1 import extractors


def _get_assessed_tasks(latest_grades, issues, components):
    '''
    latest_grades:
    | entity_id | action_by | action | graded_by | grade |
    |-----------+-----------+--------+-----------+-------|
    | ...       | ...       | ...    | ...       | ...   |

    issues:
    | key | components | ... |
    |-----+------------+-----|
    | ... | [id, ...]  | ... |

    components:
    | id  | name | ... |
    |-----+------+-----|
    | ... | ...  | ... |

    result:
    | task_id                                | action_by | action | graded_by | grade | lat  | lon  |
    |----------------------------------------+-----------+--------+-----------+-------+------+------|
    | assessment/tracker/<queue>/<component> | ...       | ...    | ...       | ...   | None | None |
    '''
    return add_tracker_task_id(latest_grades, issues, components).project(
        extractors.all(exclude='entity_id'),
        lat=extractors.const(None),
        lon=extractors.const(None)
    )


def _get_paid_tasks(assessment_log):
    '''
    assessment_log:
    | puid | task_id | ... |
    |------+---------+-----|
    | ...  | ...     | ... |

    result:
    | task_id | action_by | action | graded_by | grade | lat  | lon  |
    |---------+-----------+--------+-----------+-------+------+------|
    | ...     | puid      | close  | None      | None  | None | None |
    '''
    return assessment_log.project(
        task_id='task_id',
        action_by='puid',
        action=extractors.const(b'close'),
        graded_by=extractors.const(None),
        grade=extractors.const(None),
        lat=extractors.const(None),
        lon=extractors.const(None),
    )


def get_tasks(job, latest_grades, issues, components, assessment_log):
    '''
    latest_grades:
    | entity_id | action_by | action | graded_by | grade |
    |-----------+-----------+--------+-----------+-------|
    | ...       | ...       | ...    | ...       | ...   |

    issues:
    | key | components | ... |
    |-----+------------+-----|
    | ... | [id, ...]  | ... |

    components:
    | id  | name | ... |
    |-----+------+-----|
    | ... | ...  | ... |

    assessment_log:
    | puid | task_id | ... |
    |------+---------+-----|
    | ...  | ...     | ... |

    result:
    | task_id                                | action_by | action | graded_by | grade | lat  | lon  |
    |----------------------------------------+-----------+--------+-----------+-------+------+------|
    | Assessed tasks:                        |           |        |           |       |      |      |
    | assessment/tracker/<queue>/<component> | action_by | action | graded_by | grade | None | None |
    | ...                                    | ...       | ...    | ...       | ...   | ...  | ...  |
    | Paid tasks:                            |           |        |           |       |      |      |
    | task_id                                | puid      | close  | None      | None  | None | None |
    | ...                                    | ...       | ...    | ...       | ...   | ...  | ...  |
    '''
    return util.concat_assessed_and_paid_tasks(
        job,
        _get_assessed_tasks(latest_grades, issues, components),
        _get_paid_tasks(assessment_log)
    )
