from maps.wikimap.stat.assessment.report.lib import report

from maps.wikimap.stat.libs.common.lib.geobase_region import EARTH_REGION_ID
from maps.wikimap.stat.libs import nile_ut

from nile.api.v1 import Record
from yt.yson import to_yson_type

import pytest


def geobase_region_id_mock(lon, lat):
    if lon is None or lat is None:
        return EARTH_REGION_ID
    return lon + lat


@pytest.fixture
def geobase_fixture(monkeypatch):
    monkeypatch.setattr(
        'maps.wikimap.stat.assessment.report.lib.report.GEOBASE_FILES',
        []
    )
    monkeypatch.setattr(
        'maps.wikimap.stat.assessment.report.lib.report.geobase_region_id',
        geobase_region_id_mock
    )


def test_should_generate_tree_names():
    result = report._tree_names_for_unknown_values(tree_name_col='tree_name', value_col='value')([
        Record(tree_name=b'tree: 1', value=1),
        Record(tree_name=None,       value=2),
        Record(tree_name=b'tree: 3', value=3),
        Record(tree_name=None,       value=b'str'),
    ])
    assert sorted([
        Record(tree_name=b'tree: 1',               value=1),
        Record(tree_name=b'\tall\t',               value=2),
        Record(tree_name=b'\tall\tunknown\t',      value=2),
        Record(tree_name=b'\tall\tunknown\t2\t',   value=2),
        Record(tree_name=b'tree: 3',               value=3),
        Record(tree_name=b'\tall\t',               value=b'str'),
        Record(tree_name=b'\tall\tunknown\t',      value=b'str'),
        Record(tree_name=b'\tall\tunknown\tstr\t', value=b'str'),
    ]) == sorted(result)


def test_should_get_latest_grades_for_date():
    result = nile_ut.yt_run(
        report._get_latest_grades,
        '2020-07-28',
        assessment_unit_table=nile_ut.Table([
            Record(unit_id=1, entity_id=b'1', entity_domain=b'domain', action_by=1, action=b'action 1', test_column=1),
        ]),
        assessment_grade_table=nile_ut.Table([
            Record(unit_id=1, graded_by=11, graded_at=b'2020-07-27 00:00:00+00:00', value=b'value 11', test_column=11),
            Record(unit_id=1, graded_by=12, graded_at=b'2020-07-28 00:00:00+00:00', value=b'value 12', test_column=12),
            Record(unit_id=1, graded_by=13, graded_at=b'2020-07-29 00:00:00+00:00', value=b'value 13', test_column=13),
        ])
    )

    assert sorted([
        Record(entity_id=b'1', entity_domain=b'domain', action_by=1, action=b'action 1', graded_by=12, grade=b'value 12'),
    ]) == sorted(result)


def test_should_get_latest_grades():
    result = nile_ut.yt_run(
        report._get_latest_grades,
        '2020-07-28',
        assessment_unit_table=nile_ut.Table([
            Record(unit_id=1, entity_id=b'1', entity_domain=b'domain', action_by=1, action=b'action 1', test_column=1),
        ]),
        assessment_grade_table=nile_ut.Table([
            Record(unit_id=1, graded_by=11, graded_at=b'2020-07-28 01:01:00+00:00', value=b'value 11', test_column=11),
            Record(unit_id=1, graded_by=13, graded_at=b'2020-07-28 03:03:02+00:00', value=b'value 13', test_column=13),
            Record(unit_id=1, graded_by=12, graded_at=b'2020-07-28 02:02:01+00:00', value=b'value 12', test_column=12),
        ])
    )

    assert sorted([
        Record(entity_id=b'1', entity_domain=b'domain', action_by=1, action=b'action 1', graded_by=13, grade=b'value 13'),
    ]) == sorted(result)


def test_should_split_latest_grades_by_domain():
    result = nile_ut.yt_run(
        report._split_latest_grades_by_domain,
        latest_grades=nile_ut.Table([
            Record(entity_id=b'1', entity_domain=b'unknown',    action_by=1, action=b'action 1', graded_by=1, grade=b'grade 1'),
            Record(entity_id=b'2', entity_domain=b'feedback',   action_by=2, action=b'action 2', graded_by=2, grade=b'grade 2'),
            Record(entity_id=b'3', entity_domain=b'moderation', action_by=3, action=b'action 3', graded_by=3, grade=b'grade 3'),
            Record(entity_id=b'4', entity_domain=b'feedback',   action_by=4, action=b'action 4', graded_by=4, grade=b'grade 4'),
            Record(entity_id=b'5', entity_domain=b'edits',      action_by=5, action=b'action 5', graded_by=5, grade=b'grade 5'),
            Record(entity_id=b'6', entity_domain=b'tracker',    action_by=6, action=b'action 6', graded_by=6, grade=b'grade 6'),
        ])
    )

    assert len(result) == 4
    feedback, moderation, edits, tracker = result

    assert sorted([
        Record(entity_id=b'2', action_by=2, action=b'action 2', graded_by=2, grade=b'grade 2'),
        Record(entity_id=b'4', action_by=4, action=b'action 4', graded_by=4, grade=b'grade 4'),
    ]) == sorted(feedback)
    assert sorted([
        Record(entity_id=b'3', action_by=3, action=b'action 3', graded_by=3, grade=b'grade 3'),
    ]) == sorted(moderation)
    assert sorted([
        Record(entity_id=b'5', action_by=5, action=b'action 5', graded_by=5, grade=b'grade 5'),
    ]) == sorted(edits)
    assert sorted([
        Record(entity_id=b'6', action_by=6, action=b'action 6', graded_by=6, grade=b'grade 6'),
    ]) == sorted(tracker)


def test_should_convert_lat_lon_to_region_id(geobase_fixture):
    result = nile_ut.yt_run(
        report._lat_lon_to_region_id,
        tasks=nile_ut.Table([
            Record(grade=b'grade 1', lat=11,   lon=12,   task_id=b'task id 1', action_by=1, action=b'action 1', graded_by=11),
            Record(grade=b'grade 2', lat=None, lon=22,   task_id=b'task id 2', action_by=2, action=b'action 2', graded_by=22),
            Record(grade=b'grade 3', lat=31,   lon=32,   task_id=b'task id 3', action_by=3, action=b'action 3', graded_by=33),
            Record(grade=b'grade 4', lat=41,   lon=None, task_id=b'task id 4', action_by=4, action=b'action 4', graded_by=44),
        ])
    )

    assert sorted([
        Record(region_id=23,              task_id=b'task id 1', action_by=1, action=b'action 1', graded_by=11, grade=b'grade 1'),
        Record(region_id=EARTH_REGION_ID, task_id=b'task id 2', action_by=2, action=b'action 2', graded_by=22, grade=b'grade 2'),
        Record(region_id=63,              task_id=b'task id 3', action_by=3, action=b'action 3', graded_by=33, grade=b'grade 3'),
        Record(region_id=EARTH_REGION_ID, task_id=b'task id 4', action_by=4, action=b'action 4', graded_by=44, grade=b'grade 4'),
    ]) == sorted(result)


@pytest.mark.skip(reason='function is copied from maps/wikimap/stat/tasks_payment/report')
def test_should_convert_region_id_to_region_name_tree():
    pass


def test_should_convert_task_id_to_task_name_tree():
    result = nile_ut.yt_run(
        report._task_id_to_task_name_tree,
        tasks=nile_ut.Table([
            Record(region_name_tree=b'region 1', task_id=b'task id 1', action_by=1, action=b'action 1', graded_by=11, grade=b'grade 1'),
            Record(region_name_tree=b'region 2', task_id=b'task id 2', action_by=2, action=b'action 2', graded_by=22, grade=b'grade 2'),
            Record(region_name_tree=b'region 3', task_id=b'task id 1', action_by=3, action=b'action 3', graded_by=33, grade=b'grade 3'),
        ]),
        task_tariff_map=nile_ut.Table([
            Record(task_id=b'task id 1', task_name_tree=b'all',                 seconds_per_task=1),
            Record(task_id=b'task id 1', task_name_tree=b'all/tariff 1',        seconds_per_task=1),
            Record(task_id=b'task id 2', task_name_tree=b'all',                 seconds_per_task=2),
            Record(task_id=b'task id 2', task_name_tree=b'all/tariff 2',        seconds_per_task=2),
            Record(task_id=b'task id 2', task_name_tree=b'all/tariff 2/task 2', seconds_per_task=2),
        ])
    )

    assert sorted([
        Record(region_name_tree=b'region 1', task_name_tree=b'all',                 action_by=1, action=b'action 1', graded_by=11, grade=b'grade 1'),
        Record(region_name_tree=b'region 1', task_name_tree=b'all/tariff 1',        action_by=1, action=b'action 1', graded_by=11, grade=b'grade 1'),

        Record(region_name_tree=b'region 2', task_name_tree=b'all',                 action_by=2, action=b'action 2', graded_by=22, grade=b'grade 2'),
        Record(region_name_tree=b'region 2', task_name_tree=b'all/tariff 2',        action_by=2, action=b'action 2', graded_by=22, grade=b'grade 2'),
        Record(region_name_tree=b'region 2', task_name_tree=b'all/tariff 2/task 2', action_by=2, action=b'action 2', graded_by=22, grade=b'grade 2'),

        Record(region_name_tree=b'region 3', task_name_tree=b'all',                 action_by=3, action=b'action 3', graded_by=33, grade=b'grade 3'),
        Record(region_name_tree=b'region 3', task_name_tree=b'all/tariff 1',        action_by=3, action=b'action 3', graded_by=33, grade=b'grade 3'),
    ]) == sorted(result)


def test_should_convert_action_by_to_tree():
    result = nile_ut.yt_run(
        report._action_by_to_tree,
        tasks=nile_ut.Table([
            Record(region_name_tree=b'region 1', task_name_tree=b'task 1', action_by=1, action=b'action 1', graded_by=11, grade=b'grade 1'),
            Record(region_name_tree=b'region 2', task_name_tree=b'task 2', action_by=2, action=b'action 2', graded_by=22, grade=b'grade 2'),
        ]),
        puid_map=nile_ut.Table([
            Record(puid=1, puid_tree=b'all'),
            Record(puid=1, puid_tree=b'all/user 1'),
            Record(puid=2, puid_tree=b'all'),
            Record(puid=2, puid_tree=b'all/user 2'),
        ])
    )

    assert sorted([
        Record(region_name_tree=b'region 1', task_name_tree=b'task 1', action_by_tree=b'all',        action=b'action 1', graded_by=11, grade=b'grade 1'),
        Record(region_name_tree=b'region 1', task_name_tree=b'task 1', action_by_tree=b'all/user 1', action=b'action 1', graded_by=11, grade=b'grade 1'),
        Record(region_name_tree=b'region 2', task_name_tree=b'task 2', action_by_tree=b'all',        action=b'action 2', graded_by=22, grade=b'grade 2'),
        Record(region_name_tree=b'region 2', task_name_tree=b'task 2', action_by_tree=b'all/user 2', action=b'action 2', graded_by=22, grade=b'grade 2'),
    ]) == sorted(result)


def test_should_convert_unknown_action_by_to_tree():
    result = nile_ut.yt_run(
        report._action_by_to_tree,
        tasks=nile_ut.Table([
            Record(region_name_tree=b'region', task_name_tree=b'task', action_by=1, action=b'action', graded_by=11, grade=b'grade'),
        ]),
        puid_map=nile_ut.Table([])
    )

    assert sorted([
        Record(region_name_tree=b'region', task_name_tree=b'task', action_by_tree=b'\tall\t',             action=b'action', graded_by=11, grade=b'grade'),
        Record(region_name_tree=b'region', task_name_tree=b'task', action_by_tree=b'\tall\tunknown\t',    action=b'action', graded_by=11, grade=b'grade'),
        Record(region_name_tree=b'region', task_name_tree=b'task', action_by_tree=b'\tall\tunknown\t1\t', action=b'action', graded_by=11, grade=b'grade'),
    ]) == sorted(result)


def test_should_convert_action_to_tree():
    result = nile_ut.yt_run(
        report._action_to_tree,
        tasks=nile_ut.Table([
            Record(region_name_tree=b'region 1', task_name_tree=b'task 1', action_by_tree=b'action by 1', action=b'action 1', graded_by=1, grade=b'grade 1'),
            Record(region_name_tree=b'region 2', task_name_tree=b'task 2', action_by_tree=b'action by 2', action=b'action 2', graded_by=2, grade=b'grade 2'),
        ])
    )

    assert sorted([
        Record(region_name_tree=b'region 1', task_name_tree=b'task 1', action_by_tree=b'action by 1', action_tree=b'\tall\t',           graded_by=1, grade=b'grade 1'),
        Record(region_name_tree=b'region 1', task_name_tree=b'task 1', action_by_tree=b'action by 1', action_tree=b'\tall\taction 1\t', graded_by=1, grade=b'grade 1'),
        Record(region_name_tree=b'region 2', task_name_tree=b'task 2', action_by_tree=b'action by 2', action_tree=b'\tall\t',           graded_by=2, grade=b'grade 2'),
        Record(region_name_tree=b'region 2', task_name_tree=b'task 2', action_by_tree=b'action by 2', action_tree=b'\tall\taction 2\t', graded_by=2, grade=b'grade 2'),
    ]) == sorted(result)


def test_should_convert_graded_by_to_tree():
    result = nile_ut.yt_run(
        report._graded_by_to_tree,
        tasks=nile_ut.Table([
            Record(region_name_tree=b'region 1', task_name_tree=b'task 1', action_by_tree=b'action by 1', action_tree=b'action 1', graded_by=1, grade=b'grade 1'),
            Record(region_name_tree=b'region 2', task_name_tree=b'task 2', action_by_tree=b'action by 2', action_tree=b'action 2', graded_by=2, grade=b'grade 2'),
        ]),
        puid_map=nile_ut.Table([
            Record(puid=1, puid_tree=b'all'),
            Record(puid=1, puid_tree=b'all/user 1'),
            Record(puid=2, puid_tree=b'all'),
            Record(puid=2, puid_tree=b'all/user 2'),
        ])
    )

    assert sorted([
        Record(region_name_tree=b'region 1', task_name_tree=b'task 1', action_by_tree=b'action by 1', action_tree=b'action 1', graded_by_tree=b'all',        grade=b'grade 1'),
        Record(region_name_tree=b'region 1', task_name_tree=b'task 1', action_by_tree=b'action by 1', action_tree=b'action 1', graded_by_tree=b'all/user 1', grade=b'grade 1'),
        Record(region_name_tree=b'region 2', task_name_tree=b'task 2', action_by_tree=b'action by 2', action_tree=b'action 2', graded_by_tree=b'all',        grade=b'grade 2'),
        Record(region_name_tree=b'region 2', task_name_tree=b'task 2', action_by_tree=b'action by 2', action_tree=b'action 2', graded_by_tree=b'all/user 2', grade=b'grade 2'),
    ]) == sorted(result)


def test_should_convert_unknown_graded_by_to_tree():
    result = nile_ut.yt_run(
        report._graded_by_to_tree,
        tasks=nile_ut.Table([
            Record(region_name_tree=b'region', task_name_tree=b'task', action_by_tree=b'action by', action_tree=b'action', graded_by=1, grade=b'grade'),
        ]),
        puid_map=nile_ut.Table([])
    )

    assert sorted([
        Record(region_name_tree=b'region', task_name_tree=b'task', action_by_tree=b'action by', action_tree=b'action', graded_by_tree=b'\tall\t',             grade=b'grade'),
        Record(region_name_tree=b'region', task_name_tree=b'task', action_by_tree=b'action by', action_tree=b'action', graded_by_tree=b'\tall\tunknown\t',    grade=b'grade'),
        Record(region_name_tree=b'region', task_name_tree=b'task', action_by_tree=b'action by', action_tree=b'action', graded_by_tree=b'\tall\tunknown\t1\t', grade=b'grade'),
    ]) == sorted(result)


def test_should_convert_absent_graded_by_to_root_tree():
    result = nile_ut.yt_run(
        report._graded_by_to_tree,
        tasks=nile_ut.Table([
            Record(region_name_tree=b'region', task_name_tree=b'task', action_by_tree=b'action by', action_tree=b'action', graded_by=None, grade=None),
        ]),
        puid_map=nile_ut.Table([])
    )

    assert sorted([
        Record(region_name_tree=b'region', task_name_tree=b'task', action_by_tree=b'action by', action_tree=b'action', graded_by_tree=b'\tall\t', grade=None),
    ]) == sorted(result)


def test_should_convert_grade():
    result = nile_ut.yt_run(
        report._convert_grade,
        tasks=nile_ut.Table([
            Record(action_by_tree=b'action by 1', action_tree=b'action 1', region_name_tree=b'region 1', task_name_tree=b'task 1', graded_by_tree=b'graded by 1', grade=b'correct'),
            Record(action_by_tree=b'action by 2', action_tree=b'action 2', region_name_tree=b'region 2', task_name_tree=b'task 2', graded_by_tree=b'graded by 2', grade=b'incorrect'),
            Record(action_by_tree=b'action by 3', action_tree=b'action 3', region_name_tree=b'region 3', task_name_tree=b'task 3', graded_by_tree=b'graded by 3', grade=None),
        ])
    )

    assert sorted([
        Record(action_by_tree=b'action by 1', action_tree=b'action 1', region_name_tree=b'region 1', task_name_tree=b'task 1', graded_by_tree=b'graded by 1', correct=1, incorrect=0, quantity=0),
        Record(action_by_tree=b'action by 2', action_tree=b'action 2', region_name_tree=b'region 2', task_name_tree=b'task 2', graded_by_tree=b'graded by 2', correct=0, incorrect=1, quantity=0),
        Record(action_by_tree=b'action by 3', action_tree=b'action 3', region_name_tree=b'region 3', task_name_tree=b'task 3', graded_by_tree=b'graded by 3', correct=0, incorrect=0, quantity=1),
    ]) == sorted(result)


def test_should_aggregate_tasks_by_action_by_tree():
    result = nile_ut.yt_run(
        report._aggregate_tasks,
        tasks=nile_ut.Table([
            Record(action_by_tree=b'action by 1', action_tree=b'action', region_name_tree=b'region', task_name_tree=b'task', graded_by_tree=b'graded by', correct=1, incorrect=0,  quantity=0),
            Record(action_by_tree=b'action by 2', action_tree=b'action', region_name_tree=b'region', task_name_tree=b'task', graded_by_tree=b'graded by', correct=0, incorrect=10, quantity=0),
            Record(action_by_tree=b'action by 1', action_tree=b'action', region_name_tree=b'region', task_name_tree=b'task', graded_by_tree=b'graded by', correct=0, incorrect=0,  quantity=100),
        ]),
        isodate_str='2020-07-28'
    )

    assert sorted([
        Record(
            fielddate=b'2020-07-28', action_by_tree=b'action by 1', action_tree=b'action', region_name_tree=b'region', task_name_tree=b'task', graded_by_tree=b'graded by',
            correct_total=1, incorrect_total=0, quantity_total=100
        ),
        Record(
            fielddate=b'2020-07-28', action_by_tree=b'action by 2', action_tree=b'action', region_name_tree=b'region', task_name_tree=b'task', graded_by_tree=b'graded by',
            correct_total=0, incorrect_total=10, quantity_total=0
        ),
    ]) == sorted(result)


def test_should_aggregate_tasks_by_action_tree():
    result = nile_ut.yt_run(
        report._aggregate_tasks,
        tasks=nile_ut.Table([
            Record(action_by_tree=b'action by', action_tree=b'action 1', region_name_tree=b'region', task_name_tree=b'task', graded_by_tree=b'graded by', correct=1, incorrect=0,  quantity=0),
            Record(action_by_tree=b'action by', action_tree=b'action 2', region_name_tree=b'region', task_name_tree=b'task', graded_by_tree=b'graded by', correct=0, incorrect=10, quantity=0),
            Record(action_by_tree=b'action by', action_tree=b'action 2', region_name_tree=b'region', task_name_tree=b'task', graded_by_tree=b'graded by', correct=0, incorrect=0,  quantity=100),
        ]),
        isodate_str='2020-07-28'
    )

    assert sorted([
        Record(
            fielddate=b'2020-07-28', action_by_tree=b'action by', action_tree=b'action 1', region_name_tree=b'region', task_name_tree=b'task', graded_by_tree=b'graded by',
            correct_total=1, incorrect_total=0, quantity_total=0
        ),
        Record(
            fielddate=b'2020-07-28', action_by_tree=b'action by', action_tree=b'action 2', region_name_tree=b'region', task_name_tree=b'task', graded_by_tree=b'graded by',
            correct_total=0, incorrect_total=10, quantity_total=100
        ),
    ]) == sorted(result)


def test_should_aggregate_tasks_by_region_name_tree():
    result = nile_ut.yt_run(
        report._aggregate_tasks,
        tasks=nile_ut.Table([
            Record(action_by_tree=b'action by', action_tree=b'action', region_name_tree=b'region 1', task_name_tree=b'task', graded_by_tree=b'graded by', correct=1, incorrect=0,  quantity=0),
            Record(action_by_tree=b'action by', action_tree=b'action', region_name_tree=b'region 2', task_name_tree=b'task', graded_by_tree=b'graded by', correct=0, incorrect=10, quantity=0),
            Record(action_by_tree=b'action by', action_tree=b'action', region_name_tree=b'region 2', task_name_tree=b'task', graded_by_tree=b'graded by', correct=0, incorrect=0,  quantity=100),
        ]),
        isodate_str='2020-07-28'
    )

    assert sorted([
        Record(
            fielddate=b'2020-07-28', action_by_tree=b'action by', action_tree=b'action', region_name_tree=b'region 1', task_name_tree=b'task', graded_by_tree=b'graded by',
            correct_total=1, incorrect_total=0, quantity_total=0
        ),
        Record(
            fielddate=b'2020-07-28', action_by_tree=b'action by', action_tree=b'action', region_name_tree=b'region 2', task_name_tree=b'task', graded_by_tree=b'graded by',
            correct_total=0, incorrect_total=10, quantity_total=100
        ),
    ]) == sorted(result)


def test_should_aggregate_tasks_by_task_name_tree():
    result = nile_ut.yt_run(
        report._aggregate_tasks,
        tasks=nile_ut.Table([
            Record(action_by_tree=b'action by', action_tree=b'action', region_name_tree=b'region', task_name_tree=b'task 1', graded_by_tree=b'graded by', correct=1, incorrect=0,  quantity=0),
            Record(action_by_tree=b'action by', action_tree=b'action', region_name_tree=b'region', task_name_tree=b'task 1', graded_by_tree=b'graded by', correct=0, incorrect=10, quantity=0),
            Record(action_by_tree=b'action by', action_tree=b'action', region_name_tree=b'region', task_name_tree=b'task 2', graded_by_tree=b'graded by', correct=0, incorrect=0,  quantity=100),
        ]),
        isodate_str='2020-07-28'
    )

    assert sorted([
        Record(
            fielddate=b'2020-07-28', action_by_tree=b'action by', action_tree=b'action', region_name_tree=b'region', task_name_tree=b'task 1', graded_by_tree=b'graded by',
            correct_total=1, incorrect_total=10, quantity_total=0
        ),
        Record(
            fielddate=b'2020-07-28', action_by_tree=b'action by', action_tree=b'action', region_name_tree=b'region', task_name_tree=b'task 2', graded_by_tree=b'graded by',
            correct_total=0, incorrect_total=0, quantity_total=100
        ),
    ]) == sorted(result)


def test_should_aggregate_tasks_by_graded_by_tree():
    result = nile_ut.yt_run(
        report._aggregate_tasks,
        tasks=nile_ut.Table([
            Record(action_by_tree=b'action by', action_tree=b'action', region_name_tree=b'region', task_name_tree=b'task', graded_by_tree=b'graded by 1', correct=1, incorrect=0,  quantity=0),
            Record(action_by_tree=b'action by', action_tree=b'action', region_name_tree=b'region', task_name_tree=b'task', graded_by_tree=b'graded by 2', correct=0, incorrect=10, quantity=0),
            Record(action_by_tree=b'action by', action_tree=b'action', region_name_tree=b'region', task_name_tree=b'task', graded_by_tree=b'graded by 1', correct=0, incorrect=0,  quantity=100),
        ]),
        isodate_str='2020-07-28'
    )

    assert sorted([
        Record(
            fielddate=b'2020-07-28', action_by_tree=b'action by', action_tree=b'action', region_name_tree=b'region', task_name_tree=b'task', graded_by_tree=b'graded by 1',
            correct_total=1, incorrect_total=0, quantity_total=100
        ),
        Record(
            fielddate=b'2020-07-28', action_by_tree=b'action by', action_tree=b'action', region_name_tree=b'region', task_name_tree=b'task', graded_by_tree=b'graded by 2',
            correct_total=0, incorrect_total=10, quantity_total=0
        ),
    ]) == sorted(result)


def test_should_prepare_report_feedback_tasks_only(geobase_fixture):
    result = nile_ut.yt_run(
        report.prepare_report,
        job=nile_ut.Job(),
        date='2020-07-28',
        assessment_unit_table=nile_ut.Table([
            Record(unit_id=1, entity_id=b'1', entity_domain=b'feedback', action_by=1, action=b'accept'),
        ]),
        assessment_grade_table=nile_ut.Table([
            Record(unit_id=1, graded_by=2, graded_at=b'2020-07-28 01:01:01+01:01', value=b'correct'),
        ]),
        feedback_table=nile_ut.Table([
            Record(id=1, position=[1, 10], source=b'source', type=b'type', history=to_yson_type(
                [{b'operation': b'accept', b'modifiedBy': 1, b'modifiedAt': b'2020-07-28 01:01:01+01:01'}]
            ))
        ]),
        commit_event_table=nile_ut.Table([]),
        moderation_log=nile_ut.Table([]),
        task_active_table=nile_ut.Table([]),
        task_closed_table=nile_ut.Table([]),
        major_regions_map=nile_ut.Table([
            Record(region_id=b'11', region_tree=b'region'),
        ]),
        task_tariff_map=nile_ut.Table([
            Record(task_id=b'feedback/accept/source/type', task_name_tree=b'fb task'),
        ]),
        puid_map=nile_ut.Table([
            Record(puid=1, puid_tree=b'\tall\tuser 1\t'),
            Record(puid=2, puid_tree=b'\tall\tuser 2\t'),
        ]),
        issues=nile_ut.Table([]),
        components=nile_ut.Table([]),
        assessment_log=nile_ut.Table([]),
    )

    assert sorted([
        Record(
            fielddate=b'2020-07-28', action_by_tree=b'\tall\tuser 1\t', action_tree=b'\tall\t', region_name_tree=b'region', task_name_tree=b'fb task', graded_by_tree=b'\tall\t',
            correct_total=0, incorrect_total=0, quantity_total=1
        ),
        Record(
            fielddate=b'2020-07-28', action_by_tree=b'\tall\tuser 1\t', action_tree=b'\tall\taccept\t', region_name_tree=b'region', task_name_tree=b'fb task', graded_by_tree=b'\tall\t',
            correct_total=0, incorrect_total=0, quantity_total=1
        ),
        Record(
            fielddate=b'2020-07-28', action_by_tree=b'\tall\tuser 1\t', action_tree=b'\tall\t', region_name_tree=b'region', task_name_tree=b'fb task', graded_by_tree=b'\tall\tuser 2\t',
            correct_total=1, incorrect_total=0, quantity_total=0
        ),
        Record(
            fielddate=b'2020-07-28', action_by_tree=b'\tall\tuser 1\t', action_tree=b'\tall\taccept\t', region_name_tree=b'region', task_name_tree=b'fb task', graded_by_tree=b'\tall\tuser 2\t',
            correct_total=1, incorrect_total=0, quantity_total=0
        ),
    ]) == sorted(result)


def test_should_prepare_report_moderation_tasks_only(geobase_fixture):
    result = nile_ut.yt_run(
        report.prepare_report,
        job=nile_ut.Job(),
        date='2020-07-28',
        assessment_unit_table=nile_ut.Table([
            Record(unit_id=1, entity_id=b'1', entity_domain=b'moderation', action_by=1, action=b'resolve'),
        ]),
        assessment_grade_table=nile_ut.Table([
            Record(unit_id=1, graded_by=2, graded_at=b'2020-07-28 02:02:02+02:02', value=b'incorrect'),
        ]),
        feedback_table=nile_ut.Table([]),
        commit_event_table=nile_ut.Table([
            Record(event_id=1, bounds_geom=b'geom', primary_object_category_id=b'category')
        ]),
        moderation_log=nile_ut.Table([
            Record(puid=1, task_id=b'moderation/resolved/category', lat_min=1, lon_min=2, lat_max=3, lon_max=4),
        ]),
        task_active_table=nile_ut.Table([]),
        task_closed_table=nile_ut.Table([]),
        major_regions_map=nile_ut.Table([
            Record(region_id=b'11', region_tree=b'region'),
        ]),
        task_tariff_map=nile_ut.Table([
            Record(task_id=b'moderation/resolved/category', task_name_tree=b'mod task'),
        ]),
        puid_map=nile_ut.Table([
            Record(puid=1, puid_tree=b'\tall\tuser 1\t'),
            Record(puid=2, puid_tree=b'\tall\tuser 2\t'),
        ]),
        issues=nile_ut.Table([]),
        components=nile_ut.Table([]),
        assessment_log=nile_ut.Table([]),
    )

    assert sorted([
        Record(
            fielddate=b'2020-07-28', action_by_tree=b'\tall\tuser 1\t', action_tree=b'\tall\t', region_name_tree=b'\t10000\t', task_name_tree=b'mod task', graded_by_tree=b'\tall\t',
            correct_total=0, incorrect_total=0, quantity_total=1
        ),
        Record(
            fielddate=b'2020-07-28', action_by_tree=b'\tall\tuser 1\t', action_tree=b'\tall\tresolve\t', region_name_tree=b'\t10000\t', task_name_tree=b'mod task', graded_by_tree=b'\tall\t',
            correct_total=0, incorrect_total=0, quantity_total=1
        ),
        Record(
            fielddate=b'2020-07-28', action_by_tree=b'\tall\tuser 1\t', action_tree=b'\tall\t', region_name_tree=b'\t10000\t', task_name_tree=b'mod task', graded_by_tree=b'\tall\tuser 2\t',
            correct_total=0, incorrect_total=1, quantity_total=0
        ),
        Record(
            fielddate=b'2020-07-28', action_by_tree=b'\tall\tuser 1\t', action_tree=b'\tall\tresolve\t', region_name_tree=b'\t10000\t', task_name_tree=b'mod task', graded_by_tree=b'\tall\tuser 2\t',
            correct_total=0, incorrect_total=1, quantity_total=0
        ),
    ]) == sorted(result)


def test_should_prepare_report_edit_tasks_only(geobase_fixture):
    result = nile_ut.yt_run(
        report.prepare_report,
        job=nile_ut.Job(),
        date='2020-07-28',
        assessment_unit_table=nile_ut.Table([
            Record(unit_id=1, entity_id=b'1', entity_domain=b'edits', action_by=1, action=b'object-created'),
        ]),
        assessment_grade_table=nile_ut.Table([
            Record(unit_id=1, graded_by=2, graded_at=b'2020-07-28 03:03:03+03:03', value=b'correct'),
        ]),
        feedback_table=nile_ut.Table([]),
        commit_event_table=nile_ut.Table([
            Record(event_id=1, created_by=1, created_at=b'2020-07-28', action=b'object-created', bounds_geom=b'geom', primary_object_category_id=b'category 1', type=b'edit'),
            Record(event_id=2, created_by=3, created_at=b'2020-07-28', action=b'object-created', bounds_geom=b'geom', primary_object_category_id=b'category 2', type=b'edit'),
        ]),
        moderation_log=nile_ut.Table([]),
        task_active_table=nile_ut.Table([
            Record(event_id=1),
        ]),
        task_closed_table=nile_ut.Table([
            Record(event_id=3),
        ]),
        major_regions_map=nile_ut.Table([
            Record(region_id=b'11', region_tree=b'region'),
        ]),
        task_tariff_map=nile_ut.Table([
            Record(task_id=b'edits/object-created/category 1', task_name_tree=b'edit task 1'),
            Record(task_id=b'edits/object-created/category 2', task_name_tree=b'edit task 2'),
        ]),
        puid_map=nile_ut.Table([
            Record(puid=1, puid_tree=b'\tall\tuser 1\t'),
            Record(puid=2, puid_tree=b'\tall\tuser 2\t'),
            Record(puid=3, puid_tree=b'\tall\tuser 3\t'),
        ]),
        issues=nile_ut.Table([]),
        components=nile_ut.Table([]),
        assessment_log=nile_ut.Table([]),
    )

    assert sorted([
        Record(
            fielddate=b'2020-07-28', action_by_tree=b'\tall\tuser 1\t', action_tree=b'\tall\t', region_name_tree=b'\t10000\t', task_name_tree=b'edit task 1',
            graded_by_tree=b'\tall\tuser 2\t', correct_total=1, incorrect_total=0, quantity_total=0
        ),
        Record(
            fielddate=b'2020-07-28', action_by_tree=b'\tall\tuser 1\t', action_tree=b'\tall\tobject-created\t', region_name_tree=b'\t10000\t', task_name_tree=b'edit task 1',
            graded_by_tree=b'\tall\tuser 2\t', correct_total=1, incorrect_total=0, quantity_total=0
        ),
        Record(
            fielddate=b'2020-07-28', action_by_tree=b'\tall\tuser 3\t', action_tree=b'\tall\t', region_name_tree=b'\t10000\t', task_name_tree=b'edit task 2',
            graded_by_tree=b'\tall\t', correct_total=0, incorrect_total=0, quantity_total=1
        ),
        Record(
            fielddate=b'2020-07-28', action_by_tree=b'\tall\tuser 3\t', action_tree=b'\tall\tobject-created\t', region_name_tree=b'\t10000\t', task_name_tree=b'edit task 2',
            graded_by_tree=b'\tall\t', correct_total=0, incorrect_total=0, quantity_total=1
        ),
    ]) == sorted(result)


def test_should_prepare_report_tracker_tasks_only(geobase_fixture):
    result = nile_ut.yt_run(
        report.prepare_report,
        job=nile_ut.Job(),
        date='2020-07-28',
        assessment_unit_table=nile_ut.Table([
            Record(unit_id=1, entity_id=b'queue-1', entity_domain=b'tracker', action_by=1, action=b'close'),
        ]),
        assessment_grade_table=nile_ut.Table([
            Record(unit_id=1, graded_by=2, graded_at=b'2020-07-28 03:03:03+03:03', value=b'correct'),
        ]),
        feedback_table=nile_ut.Table([]),
        commit_event_table=nile_ut.Table([]),
        moderation_log=nile_ut.Table([]),
        task_active_table=nile_ut.Table([]),
        task_closed_table=nile_ut.Table([]),
        major_regions_map=nile_ut.Table([
            Record(region_id=b'11', region_tree=b'region'),
        ]),
        task_tariff_map=nile_ut.Table([
            Record(task_id=b'assessment/tracker/queue/component 1',       task_name_tree=b'tracker task 1'),
            Record(task_id=b'assessment/tracker/other-queue/component 3', task_name_tree=b'tracker task 2'),
        ]),
        puid_map=nile_ut.Table([
            Record(puid=1, puid_tree=b'\tall\tuser 1\t'),
            Record(puid=2, puid_tree=b'\tall\tuser 2\t'),
            Record(puid=3, puid_tree=b'\tall\tuser 3\t'),
        ]),
        issues=nile_ut.Table([
            Record(key=b'queue-1', components=to_yson_type([b'component id 1'])),
        ]),
        components=nile_ut.Table([
            Record(id=b'component id 1', name=b'component 1'),
        ]),
        assessment_log=nile_ut.Table([
            Record(puid=3, task_id=b'assessment/tracker/other-queue/component 3'),
        ])
    )

    assert sorted([
        Record(
            fielddate=b'2020-07-28', action_by_tree=b'\tall\tuser 1\t', action_tree=b'\tall\t', region_name_tree=b'\t10000\t', task_name_tree=b'tracker task 1',
            graded_by_tree=b'\tall\tuser 2\t', correct_total=1, incorrect_total=0, quantity_total=0
        ),
        Record(
            fielddate=b'2020-07-28', action_by_tree=b'\tall\tuser 1\t', action_tree=b'\tall\tclose\t', region_name_tree=b'\t10000\t', task_name_tree=b'tracker task 1',
            graded_by_tree=b'\tall\tuser 2\t', correct_total=1, incorrect_total=0, quantity_total=0
        ),
        Record(
            fielddate=b'2020-07-28', action_by_tree=b'\tall\tuser 3\t', action_tree=b'\tall\t', region_name_tree=b'\t10000\t', task_name_tree=b'tracker task 2',
            graded_by_tree=b'\tall\t', correct_total=0, incorrect_total=0, quantity_total=1
        ),
        Record(
            fielddate=b'2020-07-28', action_by_tree=b'\tall\tuser 3\t', action_tree=b'\tall\tclose\t', region_name_tree=b'\t10000\t', task_name_tree=b'tracker task 2',
            graded_by_tree=b'\tall\t', correct_total=0, incorrect_total=0, quantity_total=1
        ),
    ]) == sorted(result)
