from .schema import (
    TASKS_LOG_SCHEMA,
)
from maps.wikimap.stat.libs.geobase_regions import (
    FILES as GEOBASE_FILES,
    GEOBASE_JOB_MEMORY_LIMIT_MB,
    mercator_geom_to_region_id,
)
from nile.api.v1 import (
    extractors as ne,
    datetime as nd,
    utils as nu,
    Record,
)
from qb2.api.v1 import (
    filters as qf,
    typing,
)


_MODERATION_GAP_DAYS = 14


def _subtract_moderation_gap(date):
    return nd.next_day(date, scale='daily', offset=-_MODERATION_GAP_DAYS).decode()


def _get_edits_task_id(action, category_id):
    if category_id is None:
        category_id = 'common'
    return '/'.join(('edits', action, category_id))


def get_unmoderated_edits_tasks(commit_event_table, social_task_table, min_action_date):
    '''
    commit_event_table:
    | commit_id | type | created_at | created_by | action | primary_object_category_id | bounds_geom | ... |
    |-----------+------+------------+------------+--------+----------------------------+-------------+-----|
    | ...       | ...  | ...        | ...        | ...    | ...                        | ...         | ... |

    social_task_table:
    | commit_id | created_at | ... |
    |-----------+------------+-----|
    | ...       | ...        | ... |

    result:
    | entity_id | entity_domain | action | action_at  | action_by  | task_id                      | region_id                               |
    |-----------+---------------+--------+------------+-------------------------------------------+-----------------------------------------|
    | commit_id | "edits"       | action | created_at | created_by | edits/<action>/<category_id> | get_region_id(bbox(bounds_geom).center) |
    '''

    exclude_entities = social_task_table.filter(
        qf.compare('created_at', '>=', min_action_date),
    ).project(
        entity_id='commit_id',
    )

    return commit_event_table.filter(
        qf.compare('created_at', '>=', min_action_date),
        qf.yql_custom('CAST(Yson::ConvertToString(Yson::Parse($p0)) AS Utf8) = \'edit\'', 'type'),
    ).project(
        entity_id='commit_id',
        action='action',
        action_at='created_at',
        action_by='created_by',
        category_id='primary_object_category_id',
        geom='bounds_geom',
    ).join(
        exclude_entities,
        by='entity_id',
        type='left_only',
    ).project(
        'action_by',
        'action_at',
        'action',
        entity_id=ne.custom(str, 'entity_id').with_type(typing.Unicode),
        entity_domain=ne.const('edits').with_type(typing.Unicode),
        region_id=ne.custom(mercator_geom_to_region_id, 'geom').with_type(typing.Int64),
        task_id=ne.custom(_get_edits_task_id, 'action', 'category_id').with_type(typing.Unicode),

        files=GEOBASE_FILES,
        memory_limit=GEOBASE_JOB_MEMORY_LIMIT_MB,
    )


def get_moderation_tasks(commit_event_table, social_task_table, min_action_date):
    '''
    commit_event_table:
    | commit_id | type | primary_object_category_id | bounds_geom | ... |
    |-----------+------+----------------------------+-------------+-----|
    | ...       | ...  | ...                        | ...         | ... |

    social_task_table:
    | commit_id | resolved_by | resolved_at | closed_by | closed_at | ... |
    |-----------+-------------+-------------+-----------+-----------+-----|
    | ...       | ...         | ...         | ...       | ...       | ... |

    result:
    | entity_id | entity_domain | action | action_at                | action_by                | task_id                           | region_id                               |
    |-----------+---------------+--------+--------------------------+--------------------------+-----------------------------------+-----------------------------------------|
    | commit_id | "edits"       | action | resolved_at or closed_at | resolved_by or closed_by | moderation/<action>/<category_id> | get_region_id(bbox(bounds_geom).center) |
    '''
    @nu.with_hints(output_schema=TASKS_LOG_SCHEMA)
    def mapper(records):
        for record in records:
            entity_id = record['event_id']
            region_id = mercator_geom_to_region_id(record.get('bounds_geom'))
            category_id = record.get('primary_object_category_id') or 'common'

            resolved_at = record.get('resolved_at')
            resolved_by = record.get('resolved_by')

            closed_at = record.get('closed_at')
            closed_by = record.get('closed_by')

            if resolved_at >= min_action_date:
                yield Record(
                    entity_id=str(entity_id),
                    entity_domain='moderation',
                    action_by=resolved_by,
                    action_at=resolved_at,
                    action='resolve',
                    region_id=region_id,
                    task_id='moderation/resolved/' + category_id,
                )

            if closed_at is not None and (resolved_at != closed_at or resolved_by != closed_by):
                yield Record(
                    entity_id=str(entity_id),
                    entity_domain='moderation',
                    action_by=closed_by,
                    action_at=closed_at,
                    action='close',
                    region_id=region_id,
                    task_id='moderation/closed/' + category_id,
                )

    commits = commit_event_table.filter(
        qf.compare('created_at', '>=', _subtract_moderation_gap(min_action_date)),
    ).project(
        'event_id',
        'bounds_geom',
        'primary_object_category_id'
    )

    return social_task_table.filter(
        qf.or_(
            qf.compare('resolved_at', '>=', min_action_date, default=False),  # required to filter out unresolved tasks
            qf.compare('closed_at', '>=', min_action_date, default=False),
        )
    ).project(
        'event_id',
        'resolved_at',
        'resolved_by',
        'closed_at',
        'closed_by',
    ).join(
        commits,
        by='event_id',
        type='left',
    ).map(
        mapper,
        files=GEOBASE_FILES,
        memory_limit=GEOBASE_JOB_MEMORY_LIMIT_MB,
    )
