from maps.wikimap.stat.libs.tracker import (
    convert_staff_uid_to_puid,
    filter_by_queues,
    get_closed_issues,
    get_tracker_task_id,
)
from maps.wikimap.stat.libs.geobase_regions import EARTH_REGION_ID
import nile.api.v1.extractors as ne
from qb2.api.v1 import (
    filters as qf,
    typing,
)

GRADED_QUEUES = frozenset({'GEOCONTENTFB'})


def get_tracker_tasks(
    dates,
    components_table,
    issue_events_table,
    issues_table,
    resolutions_table,
    statuses_table,
    linked_accounts,
):
    '''
    components_table:
    | id  | name | ... |
    |-----+------+-----|
    | ... | ...  | ... |

    issue_events_table:
    | issue | date (ms) | changes (yson)             | ... |
    |-------+-----------+----------------------------+-----|
    | ...   | ...       | [                          | ... |
    |       |           |   {                        |     |
    |       |           |      'field': 'fieldName', |     |
    |       |           |      'newValue': {         |     |
    |       |           |        'value': yson_value |     |
    |       |           |      },                    |     |
    |       |           |      ...                   |     |
    |       |           |   },                       |     |
    |       |           |   ...                      |     |
    |       |           | ]                          |     |

    issues_table:
    | id  | key                   | resolution | assignee | components | storyPoints | ... |
    |-----+-----------------------+------------+----------+------------+-------------+-----|
    | ... | <queue name>-<number> | ...        | ...      | ...        | ...         | ... |

    resolutions_table:
    | id  | key                           | ... |
    |-----+-------------------------------+-----|
    | ... | internal name of a resolution | ... |

    statuses_table:
    | id  | key                       | ... |
    |-----+---------------------------+-----|
    | ... | internal name of a status | ... |

    linked_accounts:
    | date | staff_uid | puid | is_primary_link | ... |
    |------+-----------+------+-----------------+-----|
    | ...  | ...       | ...  | ...             | ... |

    result:
    | entity_id             | entity_domain | action_by                 | action_at | action  | task_id | region_id       |
    |-----------------------+---------------+---------------------------+-----------+---------+---------+-----------------|
    | GEOCONTENTFB-<number> | "tracker"     | COALESCE(puid, staff_uid) | date      | "close" | ...     | EARTH_REGION_ID |
    '''
    return get_closed_issues(
        dates,
        components_table,
        issue_events_table,
        filter_by_queues(issues_table, GRADED_QUEUES),
        resolutions_table,
        statuses_table
    ).filter(
        qf.defined('staff_uid'),
        # Due to manual payments (NMAPS-12698?), salary workers don't mark themselves as assignees
        # These tickets are also excluded from samples
    ).call(
        convert_staff_uid_to_puid, linked_accounts
    ).project(
        entity_domain=ne.const('tracker').with_type(typing.Unicode),
        entity_id='ticket',
        action_by='puid',
        action_at='date',
        action=ne.const('close').with_type(typing.Unicode),
        task_id=ne.custom(get_tracker_task_id, 'entity_id', 'component').with_type(typing.Unicode),
        region_id=ne.const(EARTH_REGION_ID).with_type(typing.Int64),
    )
