from maps.wikimap.stat.libs.geobase_regions import EARTH_REGION_ID
from maps.wikimap.stat.assessment.tasks_log.lib.tracker import get_tracker_tasks
from maps.wikimap.stat.libs import nile_ut
from nile.api.v1 import Record
from qb2.api.v1 import typing
from yt.yson import to_yson_type
import datetime

_ISSUE_EVENTS_SCHEMA = {
    'issue':   typing.Unicode,
    'date':    typing.Int64,
    'changes': typing.Yson,
}


def timestamp(date_iso_str):
    return int(datetime.datetime.fromisoformat(date_iso_str).timestamp()) * 1000


def test_should_get_tracker_tasks():
    open_changes = to_yson_type([{b'field': b'status', b'newValue': {b'value': {b'id': b'status id open'}}}])
    closed_changes = to_yson_type([{b'field': b'status', b'newValue': {b'value': {b'id': b'status id closed'}}}])

    result = nile_ut.yql_run(
        get_tracker_tasks,
        dates=['2020-01-25', '2020-01-26'],
        linked_accounts=nile_ut.Table([
            Record(staff_uid=1, puid=11,  is_primary_link=True,  date='2020-01-25'),
            Record(staff_uid=2, puid=22,  is_primary_link=True,  date='2020-01-26'),
            Record(staff_uid=3, puid=33,  is_primary_link=True,  date='2020-01-25'),
            Record(staff_uid=4, puid=44,  is_primary_link=True,  date='2020-01-25'),  # Date will not match
            Record(staff_uid=5, puid=55,  is_primary_link=True,  date='2020-01-25'),
            Record(staff_uid=5, puid=555, is_primary_link=False, date='2020-01-25'),  # Is not primary link
            Record(staff_uid=6, puid=66,  is_primary_link=True,  date='2020-01-25'),
            Record(staff_uid=7, puid=77,  is_primary_link=True,  date='2020-01-25'),
            Record(staff_uid=8, puid=88,  is_primary_link=True,  date='2022-01-25'),  # Unused
        ]),
        components_table=nile_ut.Table([
            Record(id='component id 1', name='component 1'),
            Record(id='component id 2', name='component 2'),
            Record(id='component id 3', name='component 3'),
            Record(id='component id 4', name='component 4'),
            Record(id='component id 5', name='component 5'),
            Record(id='component id 6', name='component 6'),
            Record(id='component id 7', name='component 7'),
            Record(id='component id 8', name='component 8'),  # Unused
        ]),
        issue_events_table=nile_ut.Table([
            # First closed at another day
            Record(issue='issue id 1', date=timestamp('2020-01-23'), changes=closed_changes),
            Record(issue='issue id 1', date=timestamp('2020-01-24'), changes=open_changes),
            Record(issue='issue id 1', date=timestamp('2020-01-25'), changes=closed_changes),

            # Non-closed
            Record(issue='issue id 2', date=timestamp('2020-01-26'), changes=open_changes),

            # Okay, but reopened later
            Record(issue='issue id 3', date=timestamp('2020-01-25'), changes=closed_changes),
            Record(issue='issue id 3', date=timestamp('2020-01-27'), changes=open_changes),

            # Okay
            Record(issue='issue id 4', date=timestamp('2020-01-26'), changes=closed_changes),

            # Unpaid resolution
            Record(issue='issue id 5', date=timestamp('2020-01-25'), changes=closed_changes),

            # Unpaid queue
            Record(issue='issue id 6', date=timestamp('2020-01-25'), changes=closed_changes),

            # No assignee
            Record(issue='issue id 7', date=timestamp('2020-01-25'), changes=closed_changes),
        ], schema=_ISSUE_EVENTS_SCHEMA),
        issues_table=nile_ut.Table([
            Record(id='issue id 1', resolution='resolution id fixed', assignee=1,    components=to_yson_type([b'component id 1']), key='GEOCONTENTFB-1', storyPoints=1),
            Record(id='issue id 2', resolution='resolution id fixed', assignee=2,    components=to_yson_type([b'component id 2']), key='GEOCONTENTFB-2', storyPoints=2),
            Record(id='issue id 3', resolution='resolution id fixed', assignee=3,    components=to_yson_type([b'component id 3']), key='GEOCONTENTFB-3', storyPoints=3),
            Record(id='issue id 4', resolution='resolution id fixed', assignee=4,    components=to_yson_type([b'component id 4']), key='GEOCONTENTFB-4', storyPoints=4),
            Record(id='issue id 5', resolution='resolution id other', assignee=5,    components=to_yson_type([b'component id 5']), key='GEOCONTENTFB-5', storyPoints=5),
            Record(id='issue id 6', resolution='resolution id fixed', assignee=5,    components=to_yson_type([b'component id 6']), key='MAPSPW-6',       storyPoints=6),
            Record(id='issue id 7', resolution='resolution id fixed', assignee=None, components=to_yson_type([b'component id 7']), key='GEOCONTENTFB-7', storyPoints=7),
        ]),
        resolutions_table=nile_ut.Table([
            Record(id='resolution id fixed',  key='fixed'),
            Record(id='resolution id other',  key='other'),
            Record(id='resolution id unused', key='unused'),  # Unused entry
        ]),
        statuses_table=nile_ut.Table([
            Record(id='status id closed', key='closed'),
            Record(id='status id open',   key='open'),
            Record(id='status id unused', key='unused'),  # Unused entry
        ])
    )

    assert sorted([
        Record(entity_id='GEOCONTENTFB-3', entity_domain='tracker', action_by=33, action_at='2020-01-25', action='close', task_id='tracker/geocontentfb/component 3', region_id=EARTH_REGION_ID),
        Record(entity_id='GEOCONTENTFB-4', entity_domain='tracker', action_by=4,  action_at='2020-01-26', action='close', task_id='tracker/geocontentfb/component 4', region_id=EARTH_REGION_ID),
    ]) == sorted(result)
