from nile.api.v1 import (
    filters as nf,
    extractors as ne,
    aggregators as na,
    Record
)

USERS_DUMP_DIR = 'home/logfeller/logs/nmaps-users-dump-log/1d'
EDITS_LOG_DIR = 'home/logfeller/logs/nmaps-edits-log/1d'

GEOCUBE_DIR = 'home/geosearch-prod/geocube/1d'
GEOCUBE_TABLES = ('maps', 'mobile_maps', 'navi', 'serp', 'touch_maps')

ORIGIN_WHITELIST_TABLE = 'home/maps/core/nmaps/production/mailing/geocube_human_origin_white_list'

MOD_STATUSES = (b'common', b'expert', b'moderator')


def read_origin_whitelist(job):
    """Origin value is used to distinguish requests generated by users from automatic ones
    """
    whitelist_table = job.table(ORIGIN_WHITELIST_TABLE)
    return [record['origin_value'] for record in whitelist_table.read()]


def edits_tables(job, first_date_iso, last_date_iso):
    path = '{0}/{{{1}..{2}}}'.format(EDITS_LOG_DIR, first_date_iso, last_date_iso)
    return job.table(path)


def geocube_tables(job, first_date_iso, last_date_iso):
    path = '{0}/{{{1}..{2}}}/{{{3}}}'.format(
        GEOCUBE_DIR, first_date_iso, last_date_iso, ','.join(GEOCUBE_TABLES))
    return job.table(path)


def users_table(job, date_iso):
    return job.table('{0}/{1}'.format(USERS_DUMP_DIR, date_iso))


def created_addresses(edits, users):
    return edits.join(users, by='puid', type='inner').filter(
        nf.equals('action', b'object-created'),
        nf.equals('object_category', b'addr'),
        nf.custom(lambda status: status in MOD_STATUSES, 'moderation_status')
    ).project(
        'object_id',
        'puid',
        'geom',
        'iso_eventtime'
    )


def geocube_answers(geocube, origin_whitelist):
    def flatten_by_answers(records):
        for record in records:
            if record['origin'] not in origin_whitelist:
                continue
            for answer in record.answers_and_clicks:
                if answer[b'base'] == b'geocoder':
                    yield Record(record, answer=answer)

    return geocube.map(
        flatten_by_answers
    ).project(
        'serpid',
        'bounds',
        object_id=ne.custom(lambda a: a[b'id'], 'answer'),
        clicks=ne.custom(lambda a: len(a[b'clicks']), 'answer')
    )


def answers_for_addresses(addresses, answers):
    def center(geom):
        """Center of geom
        geom's format: b"[[39.841961169,57.618111987],[39.8442,57.619378]]"
        """
        x = 0.0
        y = 0.0
        coord_strings = geom.strip(b'[]').split(b'],[')
        for point in coord_strings:
            px, py = map(float, point.split(b','))
            x += px
            y += py
        return x / len(coord_strings), y / len(coord_strings)

    def is_inside_bounds_extractor(geom, bounds):
        """Check if geom's center is inside bounds.
        bounds' format: b"30.44846552,59.93876387,30.77668207,60.06205076"
        """
        if geom is None or bounds is None:
            return False
        xmin, ymin, xmax, ymax = map(float, bounds.split(b','))
        x, y = center(geom)
        return xmin < x < xmax and ymin < y < ymax

    def seen(clicks, is_inside_bounds):
        return clicks > 0 or is_inside_bounds

    return addresses.join(answers, by='object_id').project(
        ne.all(),
        is_inside_bounds=ne.custom(is_inside_bounds_extractor, 'geom', 'bounds')
    ).filter(
        nf.custom(seen, 'clicks', 'is_inside_bounds')
    ).unique('serpid', 'object_id', 'puid')


def answers_for_users(shown_addresses):
    return shown_addresses.groupby('puid').aggregate(
        shows_count=na.count(),
        shown_objects_count=na.count_distinct('object_id')
    )


def make_job(
    job, result_table,
    creation_period_start, creation_period_finish,
    show_period_start, show_period_finish
):
    origin_whitelist = read_origin_whitelist(job)

    edits = edits_tables(job, creation_period_start, creation_period_finish)
    geocube = geocube_tables(job, show_period_start, show_period_finish)
    users = users_table(job, show_period_finish)

    addresses = created_addresses(edits, users)
    answers = geocube_answers(geocube, origin_whitelist)

    shown_addresses = answers_for_addresses(addresses, answers)
    shown_addresses_users = answers_for_users(shown_addresses)

    shown_addresses_users.put(result_table)

    return job
