import yt.wrapper

from collections import namedtuple
import os
import operator
import requests


AB_TOKEN_ENV = "AB_TOKEN"
YT_TOKEN_ENV = "YT_TOKEN"

AB_TASK_URL = "https://ab.yandex-team.ru/api/task/?form=full"
AB_TESTID_URL = "https://ab.yandex-team.ru/api/testid/?form=full"

YT_PROXY = "hahn"

FEEDBACK_METRICS_META_DIR = "//home/maps/core/nmaps/analytics/feedback_metrics/meta"
TICKETS_TABLE = FEEDBACK_METRICS_META_DIR + "/experiment_ticket"
TEST_IDS_TABLE = FEEDBACK_METRICS_META_DIR + "/test_id"

DESCRIPTION = "description"
TEST_ID = "test_id"
TEST_IDS = "test_ids"
TICKET = "ticket"
TICKETS = "tickets"
TITLE = "title"


class TicketInfo:
    def __init__(self, task_json):
        self.ticket = task_json[TICKET]
        self.title = task_json[TITLE]

    def to_yt_row(self):
        return {
            TICKET: self.ticket,
            TITLE: self.title,
            DESCRIPTION: "{} ({})".format(self.title, self.ticket)
        }


class TestIdInfo:
    def __init__(self, test_id_json):
        self.test_id = test_id_json["testid"]  # differs from constant TEST_ID
        self.title = test_id_json[TITLE]
        self.ticket = test_id_json[TICKET]

    def to_yt_row(self):
        return {
            TEST_ID: str(self.test_id),
            TICKET: self.ticket,
            TITLE: self.title,
            DESCRIPTION: "{} ({})".format(self.title, self.test_id)
        }


ExperimentsInfo = namedtuple("ExperimentsInfo", ["tickets_info", "test_ids_info"])


def get_response_json(request, params, headers):
    response = requests.get(
        request,
        params=params,
        headers=headers
    )
    response.raise_for_status()
    return response.json()


def get_experiments_info(ab_token):
    headers = {
        "X-Ab-Strict-Auth": "true",
        "Authorization": "OAuth {}".format(ab_token)
    }

    aspect_tasks = [
        get_response_json(AB_TASK_URL, {"aspect": "nmaps_ui"}, headers),
        get_response_json(AB_TASK_URL, {"aspect": "fbapi"}, headers),
    ]

    test_ids = []
    tickets_info = []
    for tasks_json in aspect_tasks:
        for task_json in tasks_json:
            test_ids.extend(task_json["testids"])  # differs from constant TEST_IDS
            tickets_info.append(TicketInfo(task_json))

    test_ids_info = []
    if test_ids:
        test_ids_json = get_response_json(AB_TESTID_URL, {"id": test_ids}, headers)
        test_ids_info = [TestIdInfo(test_id_json) for test_id_json in test_ids_json]

    return ExperimentsInfo(tickets_info, test_ids_info)


def write_table_with_schema(data, table, schema):
    yt.wrapper.write_table(
        table=yt.wrapper.TablePath(
            name=table,
            schema=schema),
        input_stream=data
    )


def upload_experiments_meta(experiments_info):
    tickets = [ticket_info.to_yt_row() for ticket_info in experiments_info.tickets_info]
    write_table_with_schema(
        data=sorted(tickets, key=operator.itemgetter(TICKET)),
        table=TICKETS_TABLE,
        schema=[
            {"name": TICKET, "type": "string", "sort_order": "ascending"},
            {"name": TITLE, "type": "string"},
            {"name": DESCRIPTION, "type": "string"},
        ]
    )

    test_ids = [test_id_info.to_yt_row() for test_id_info in experiments_info.test_ids_info]
    write_table_with_schema(
        data=sorted(test_ids, key=operator.itemgetter(TEST_ID)),
        table=TEST_IDS_TABLE,
        schema=[
            {"name": TEST_ID, "type": "string", "sort_order": "ascending"},
            {"name": TICKET, "type": "string"},
            {"name": TITLE, "type": "string"},
            {"name": DESCRIPTION, "type": "string"},
        ]
    )


def update_experiments_meta(ab_token=None, yt_token=None):
    if ab_token is None:
        ab_token = os.environ[AB_TOKEN_ENV] if AB_TOKEN_ENV in os.environ else None
    if yt_token is None:
        yt_token = os.environ[YT_TOKEN_ENV] if YT_TOKEN_ENV in os.environ else None

    yt.wrapper.config["proxy"]["url"] = YT_PROXY
    yt.wrapper.config["token"] = yt_token

    upload_experiments_meta(get_experiments_info(ab_token))
