import datetime

from nile.api.v1 import (
    clusters,
    cli,
    datetime as nd,
    Job,
)
import yt.wrapper as yt

import maps.wikimap.stat.fbapi_feedback_metrics.calculate_metrics.lib.build_graph as build_graph


JOB_OPTIONS = [
    cli.Option(
        "from-date",
        help="The first date of the range, iso format: YYYY-mm-dd. Equals to yesterday by default.",
        required=False,
    ),
    cli.Option(
        "to-date",
        help="The last day of the range, iso format: YYYY-mm-dd. Equals to --from-date by default.",
        required=False,
    ),
    cli.Option(
        "result-path",
        help="Path to YT directory for storing result tables with feedback metrics.",
        required=False,
    ),
    cli.Option(
        "dump-date",
        help="Dump date of nmaps database to use during metrics calculation",
        required=False,
    ),
    cli.Option(
        "skip-bebr",
        help="Do not use bebr logs (they are available only for the last year)",
        required=False,
        is_flag=True,
    ),
]


def first_monday_before(date: str) -> str:
    return nd.round_period_weekly(date).decode("utf-8")


def first_sunday_after(date: str) -> str:
    week_start = nd.Datetime.from_iso(first_monday_before(date))
    week_end = week_start.next(offset=6)
    return week_end.date().isoformat()


def get_yesterday() -> str:
    return (datetime.date.today() - datetime.timedelta(days=1)).isoformat()


@cli.statinfra_job(options=JOB_OPTIONS)
def make_job(job: Job, options) -> Job:
    """
    Calculate feedback metrics. YT and YQL tokens are required in order to run evaluations.
    Result metrics are split into weekly output tables. Week starts on Monday.
    Weekly tables are identified by the first day of the week.
    The provided dates range [from-date, to-date] is expanded in both directions
    to start on the closest Monday and end on the closest Sunday.
    """
    cluster = clusters.yql.Hahn()
    yt.config.set_proxy(cluster.proxy)

    from_date: str = options.from_date if options.from_date else get_yesterday()
    from_date = first_monday_before(from_date)

    to_date: str = options.to_date if options.to_date else from_date
    to_date = first_sunday_after(to_date)

    dump_date: str = options.dump_date if options.dump_date else get_yesterday()

    build_graph.make_job(
        job,
        from_date,
        to_date,
        dump_date,
        options.result_path,
        windows_days=[1, 7, 28],
        use_bebr=not options.skip_bebr)
    return job


def main() -> None:
    cli.run()


if __name__ == "__main__":
    main()
