import typing as tp

from nile.api.v1 import (
    extractors as ne,
)
from maps.wikimap.stat.libs.common.lib import geobase_region

from .constants import NULL_KEY


class OriginalTask:
    def __init__(self, data: tp.Optional[tp.Dict[bytes, tp.Any]]):
        self._data = data

    def extract_region_id(self) -> tp.Optional[str]:
        # Sometimes Data Transfer from PostgreSQL to YT skips original_task field.
        # https://st.yandex-team.ru/NMAPS-14048
        if self._data is None:
            return None

        form_point = self._data[b"form_point"]
        lon, lat = float(form_point[b"lon"]), float(form_point[b"lat"])
        return str(geobase_region.geobase_region_id(lon, lat))

    def extract_metadata(self) -> tp.Optional[tp.Dict[bytes, tp.Any]]:
        if self._data is None:
            return None

        return self._data.get(b"metadata")

    def extract_client_id(self) -> str:
        metadata = self.extract_metadata()
        if metadata is None:
            return NULL_KEY

        client_id: bytes = metadata.get(b"client_id")
        return client_id.decode("utf-8") if client_id else NULL_KEY

    def extract_client_context_id(self) -> str:
        if self._data is None:
            return NULL_KEY

        client_context_id: bytes = self._data.get(b"form_context", {}).get(b"client_context_id")
        return client_context_id.decode("utf-8") if client_context_id else NULL_KEY

    def extract_some_user_id(self) -> tp.Optional[str]:
        metadata = self.extract_metadata()
        if metadata is None:
            return None

        uid: tp.Optional[int] = metadata.get(b"uid")
        if uid is not None:
            return str(uid)

        yandexuid: tp.Optional[bytes] = metadata.get(b"yandexuid")
        if yandexuid is not None:
            return yandexuid.decode("utf-8")

        device_id: tp.Optional[bytes] = metadata.get(b"device_id")
        if device_id is not None:
            return device_id.decode("utf-8")
        return None

    def extract_test_ids(self) -> tp.Optional[tp.List[str]]:
        metadata = self.extract_metadata()
        if metadata is None:
            return None

        test_ids: tp.Optional[tp.List[bytes]] = metadata.get(b"test_ids")
        if test_ids is not None:
            return [test_id.decode("utf-8") for test_id in test_ids]

    def extract(self, field: bytes) -> str:
        """
        This is a general-purpose extractor for top-level keys.
        If key is present in original_task, then it's value
        converted to str is returned. Otherwise NULL_KEY is returned.

        NULL_KEY is used to allow filtering records with empty key.
        null/empty keys cannot be displayed in Datalens selectors :(
        """
        value = self._data.get(field) if self._data else None
        return value.decode("utf-8") if value else NULL_KEY


def make_extractor(method, *args):
    def extractor(data: tp.Optional[tp.Dict[bytes, tp.Any]]):
        original_task = OriginalTask(data)
        return method(original_task, *args)
    return ne.custom(extractor, "original_task").with_type(str)
