from nile.api.v1 import (
    filters as nf,
    aggregators as na,
    extractors as ne,
    datetime as nd,
    statface as ns
)

from common import load_feedback_db_table, prepare_feedback_table, cut_to_seconds


def resolved_on_date(resolved_at, fielddate):
    return resolved_at is not None and nd.round_period_daily(cut_to_seconds(resolved_at)) == fielddate


def filter_relevant(feedback_table):
    """Account only for relevant road closures feedback

    We calculate statistics only for road closures feedback that was processed (resolved) during each day of interest.
    """
    return feedback_table.filter(nf.and_(
        nf.custom(resolved_on_date, 'resolved_at', 'fielddate'),
        nf.equals('type', 'road-closure'),
    ))


def age_minutes(created_at, resolved_at):
    created_datetime = nd.Datetime.from_iso(cut_to_seconds(created_at))
    resolved_datetime = nd.Datetime.from_iso(cut_to_seconds(resolved_at))
    return (resolved_datetime - created_datetime).total_seconds() / 60.0


def run_aggregation(feedback_table):
    relevant = filter_relevant(feedback_table).label('relevant')

    age = relevant.project(
        'fielddate',
        'workflow',
        age_minutes=ne.custom(age_minutes, 'created_at', 'resolved_at').with_type(float)
    )

    daily_agg = age.groupby('fielddate', 'workflow').aggregate(
        sum_counter=na.count(),
        max_age_minutes=na.max('age_minutes'),
        percentile_85_age_minutes=na.percentile('age_minutes', 85.0)
    ).label('daily_result')

    return daily_agg


def calculate_report(feedback_db_table, from_date_iso_str, to_date_iso_str):
    feedback_table = prepare_feedback_table(feedback_db_table, from_date_iso_str, to_date_iso_str)
    return run_aggregation(feedback_table)


def make_testable_job(job, from_date_iso_str, to_date_iso_str):
    feedback_db_table = job.table('').label('feedback_db')

    calculate_report(feedback_db_table, from_date_iso_str, to_date_iso_str)

    return job


def make_job(job, from_date_iso_str, to_date_iso_str, report_path, statface_client):
    feedback_db_table = load_feedback_db_table(job).label('feedback_db')

    daily_agg = calculate_report(feedback_db_table, from_date_iso_str, to_date_iso_str)

    report = ns.StatfaceReport()\
        .path(report_path)\
        .scale('daily')\
        .client(statface_client)
    daily_agg.publish(report, allow_change_job=True)

    return job
