from nile.api.v1 import (
    datetime as nd,
    Record
)

FEEDBACK_DB_TABLE_PATH = '//home/maps/core/nmaps/analytics/feedback/db/feedback_latest'


def load_feedback_db_table(job):
    return job.table(FEEDBACK_DB_TABLE_PATH)


def cut_to_seconds(datetime_iso_str):
    return datetime_iso_str[:19]


def calculate_state_duration(record, from_date, to_date, state):
    """Extracts from history intervals in 'open' or 'need-info' state.
       Returns:
       (summed time (in days) of intervals until to_date            - float,
        if feedback was in specified state in (from_date, to_date)  - bool,
        if feedback in specified state at the to_date moment        - bool
    """

    assert state == 'need-info' or state == 'open'

    def extract_state_from_history(record, to_date, state):
        """Returns array of intervals [<from_time>, <till_time>]
        when record is in State==state.
        """
        def normalize_state(item_state):
            """Possible scenarios depending on normalized state:
            ignore - do nothing;
            open - feedback is in 'open' state;
            need-info - feedback is in 'need-info' state;
            deploy - end of feedback life cycle, stop processing;
            resolve - terminal state.
            """
            if item_state in ('accept', 'reject'):
                return 'resolve'
            if item_state in ('reveal', 'open'):
                return 'open'
            if item_state in ('deploy', 'need-info'):
                return item_state
            else:
                return 'ignore'

        result = []
        start_datetime = None
        current_state = None
        history = record.get('history')
        resolved_datetime = None

        for item in history:
            item_state = normalize_state(item["operation"])
            if item_state == 'ignore':
                continue

            if item_state == 'deploy':
                current_state = 'deploy'
                break

            item_datetime = nd.Datetime.from_iso(cut_to_seconds(item["modifiedAt"]))
            if item_datetime >= to_date:
                break

            if item_state == state:
                if current_state is None or current_state != state:
                    start_datetime = item_datetime
                    if resolved_datetime is not None:
                        result[:] = []
                        resolved_datetime = None
            else:
                if start_datetime is not None and current_state == state:
                    result.append((start_datetime, item_datetime))
            if item_state == 'resolve':
                if item_datetime < from_date:
                    result[:] = []
                else:
                    resolved_datetime = item_datetime
            current_state = item_state

        if current_state is not None and current_state == state:
            result.append((start_datetime, to_date))
        return result

    state_intervals = extract_state_from_history(record, to_date, state)
    if not state_intervals:
        return (0.0, False, False)

    duration = 0
    for start_datetime, end_datetime in state_intervals:
        duration += (end_datetime - start_datetime).total_seconds()

    return (duration / (24 * 60 * 60.0),
            state_intervals[-1][1] > from_date,
            state_intervals[-1][1] >= to_date)


def prepare_feedback_table(feedback_db_table, from_date_iso_str, to_date_iso_str):
    date_range = list(
        nd.date_range(from_date_iso_str, to_date_iso_str, stringify=False))
    created_margins = [date.next() for date in date_range]

    def prepare_feedback_table_mapper(records):
        for record in records:
            for date, created_margin in zip(date_range, created_margins):
                age_days, open_in_window, open_at_end = calculate_state_duration(record, date, created_margin, 'open')
                age_days_need_info, need_info_in_window, need_info_at_end = calculate_state_duration(
                    record, date, created_margin, 'need-info')
                if open_in_window or need_info_in_window:
                    yield Record(
                        record,
                        fielddate=date.to_str(no_time=True),
                        open_at_end=int(open_at_end),
                        need_info_at_end=int(need_info_at_end),
                        age_days=age_days,
                        need_info_in_window=need_info_in_window,
                        age_days_need_info=age_days_need_info if age_days_need_info > 0 else None)

    feedback_table = feedback_db_table.project(
        'created_at', 'resolved_at', 'type', 'source', 'workflow', 'history'
    )

    return feedback_table.map(prepare_feedback_table_mapper)
