import datetime
import json
from library.python import resource
from nile.api.v1 import cli, datetime as nd
from maps.pylibs.utils.lib.common import require
import maps.wikimap.stat.libs.common.lib.cli_options as cli_opts

from . import constants as c


COUNT_REPORT = 'count'
LENGTH_REPORT = 'length'
REPORT_CHOICES = [COUNT_REPORT, LENGTH_REPORT]

DEFAULT_YMAPSDF_PATH = '//home/maps/core/garden/stable/ymapsdf/latest'
DEFAULT_REGIONS = list(json.loads(resource.find('regions.json')))


JOB_OPTIONS = [
    cli_opts.results_option(),
    cli.Option(
        'report-type',
        default=COUNT_REPORT,
        choices=REPORT_CHOICES,
        help='Report type, one of ({choices}), default is {count}'.format(
            choices=', '.join(REPORT_CHOICES), count=COUNT_REPORT
        )
    ),
    cli.Option(
        'ymapsdf-path',
        help='Path to ymapsdf, default is {}. Tool accepts the only today date'
        'in --dates parameter if --ymapsdf-path is not specified explicitly. '
        'Anyway --dates parameter accepts the single date because tool can '
        'calculate statistics for excatly one ymapsdf dump. '
        'Parameter --ymapsdf-path is supposed to set custom path for old data '
        'in order to recalculate statistics.'.format(DEFAULT_YMAPSDF_PATH)
    ),
    cli.Option(
        'no-regions',
        is_flag=True,
        help='Set flag to use empty list of regions instead of {}'.format(DEFAULT_REGIONS)
    ),
] + [
    cli.Option(
        table_name,
        is_flag=True,
        help='Process {table} if flag is set'.format(table=table_name)
    ) for table_name in c.ALL_AVAILABLE_TABLES
]


def get_report_descriptor(report_type):
    if report_type == COUNT_REPORT:
        return c.COUNT_REPORT_DESCRIPTOR
    elif report_type == LENGTH_REPORT:
        return c.LENGTH_REPORT_DESCRIPTOR
    else:
        raise RuntimeError('Wrong report type {0}'.format(report_type))


def extract_fielddate(options):
    fielddate = datetime.date.today().isoformat()
    options_date = cli_opts.get_single_date(options)
    if options.ymapsdf_path is not None:
        # do not use today for manually launch
        fielddate = options_date
    else:
        require(
            fielddate == options_date,
            RuntimeError('Use only today in --dates parameter without explicitly set ymapsdf path')
        )

    cur_period = nd.round_period(fielddate, options.scale)
    require(
        cur_period == fielddate,
        RuntimeError(
            'For {scale} report you must specify the first date '
            'of the corresponding period'.format(scale=options.scale)
        )
    )
    return fielddate


def extract_tables(options, available_tables):
    tables = [
        table_name for table_name in available_tables
        if getattr(options, table_name)
    ]
    # Use all available tables if no corresponding options are set
    return tables or list(available_tables)
