import common
import maps.wikimap.stat.kpi.absolute_objects_counts_stat.lib.ad_processors as ap

from nile.api.v1 import (
    local,
    Record
)
import pytest


@pytest.fixture
def ad_sources(major_regions):
    return {
        'ad': local.StreamSource([
            Record(ad_id=1, level_kind=1, p_ad_id=2),
            Record(ad_id=2, level_kind=1, p_ad_id=None),
            Record(ad_id=3, level_kind=2, p_ad_id=1),
        ]),
        'ad_face': local.StreamSource([
            Record(ad_id=1, face_id=1)
        ]),
        'locality': local.StreamSource([
            Record(ad_id=1, town=True),
            Record(ad_id=2, town=False),
        ]),
        'geocoder_source_id_map': local.StreamSource([
            Record(mutable_source_id=1, geo_id=1),  # Россия, Московская область
            Record(mutable_source_id=2, geo_id=3),  # Россия, Центральный федеральный округ
        ]),
        'major_regions': local.StreamSource(major_regions),
    }


def test_ad_with_geoid(statface_client, job, ad_sources):
    job = common.make_count_job(statface_client, job, ['ad'])
    ad_with_geoid = []
    job.local_run(
        sources=ad_sources,
        sinks={
            'ad_with_geoid': local.ListSink(ad_with_geoid),
        }
    )
    assert sorted(ad_with_geoid) == sorted([
        Record(ad_id=1, count=1, geo_id=1, level_kind=1, p_ad_id=2,    mutable_source_id=1),
        Record(ad_id=2, count=1, geo_id=3, level_kind=1, p_ad_id=None, mutable_source_id=2),
        Record(ad_id=3, count=1, geo_id=1, level_kind=2, p_ad_id=1),
    ])


def test_ad_region_and_shape(statface_client, job, ad_sources):
    job = common.make_count_job(statface_client, job, ['ad'])
    ad_with_region_and_shape = []
    job.local_run(
        sources=ad_sources,
        sinks={
            'ad_with_region_and_shape_type': local.ListSink(ad_with_region_and_shape),
        }
    )
    assert sorted(ad_with_region_and_shape) == sorted([
        Record(ad_id=1, count=1, level_kind=1, shape_type='polygon', region_id='1'),
        Record(ad_id=2, count=1, level_kind=1, shape_type='point',   region_id='3'),
        Record(ad_id=3, count=1, level_kind=2, shape_type='point',   region_id='1'),
    ])


def test_counted_ad_by_shape_type(statface_client, job, ad_sources):
    job = common.make_count_job(statface_client, job, ['ad'])

    counted_ad_by_shape_type = []
    job.local_run(
        sources=ad_sources,
        sinks={
            'counted_ad_by_shape_type': local.ListSink(counted_ad_by_shape_type),
        }
    )
    assert sorted(counted_ad_by_shape_type) == sorted([
        Record(count=2, region_tree='\t10000\t',                 shape_type='point'),    # Земля
        Record(count=1, region_tree='\t10000\t',                 shape_type='polygon'),  # Земля
        Record(count=2, region_tree='\t10000\t225\t',            shape_type='point'),    # Земля/Россия
        Record(count=1, region_tree='\t10000\t225\t',            shape_type='polygon'),  # Земля/Россия
        Record(count=2, region_tree='\t10000\t225\t3\t',         shape_type='point'),    # Земля/Россия/ЦФО
        Record(count=1, region_tree='\t10000\t225\t3\t',         shape_type='polygon'),  # Земля/Россия/ЦФО
        Record(count=1, region_tree='\t10000\t225\t3\t1\t',      shape_type='point'),    # Земля/Россия/ЦФО/Московская область
        Record(count=1, region_tree='\t10000\t225\t3\t1\t',      shape_type='polygon'),  # Земля/Россия/ЦФО/Московская область
        Record(count=2, region_tree='\t10000\t225\t999999000\t', shape_type='point'),    # Земля/Россия/0+
        Record(count=1, region_tree='\t10000\t225\t999999000\t', shape_type='polygon'),  # Земля/Россия/0+
    ])


def test_add_any_shape_ad_count_reducer():
    groups = [
        ('\t10000\t', [
            Record(count=1, region_tree='\t10000\t', shape_type='point'),
            Record(count=3, region_tree='\t10000\t', shape_type='polygon'),
        ]),
        ('\t10000\t225\t', [
            Record(count=7, region_tree='\t10000\t225\t', shape_type='point'),
        ]),
        ('\t10000\t225\t3\t', [
            Record(count=8, region_tree='\t10000\t225\t3\t', shape_type='polygon'),
        ]),
    ]

    assert sorted(list(ap.add_any_shape_ad_count_reducer(groups))) == sorted([
        Record(region_tree='\t10000\t',         count=1, shape_type='point'),
        Record(region_tree='\t10000\t',         count=3, shape_type='polygon'),
        Record(region_tree='\t10000\t',         count=4, shape_type='all'),
        Record(region_tree='\t10000\t225\t',    count=7, shape_type='point'),
        Record(region_tree='\t10000\t225\t',    count=7, shape_type='all'),
        Record(region_tree='\t10000\t225\t3\t', count=8, shape_type='polygon'),
        Record(region_tree='\t10000\t225\t3\t', count=8, shape_type='all'),
    ])


def test_ad_level_kind_count(statface_client, job, ad_sources):
    job = common.make_count_job(statface_client, job, ['ad'])

    counted_ad_level_kind = []
    job.local_run(
        sources=ad_sources,
        sinks={
            'counted_ad_by_level_kind': local.ListSink(counted_ad_level_kind),
        }
    )
    assert sorted(counted_ad_level_kind) == sorted([
        Record(category='ad level_kind=1', count=2, fielddate='2019-01-01', region_tree='\t10000\t'),                  # Земля
        Record(category='ad level_kind=1', count=2, fielddate='2019-01-01', region_tree='\t10000\t225\t'),             # Земля/Россия
        Record(category='ad level_kind=1', count=2, fielddate='2019-01-01', region_tree='\t10000\t225\t3\t'),          # Земля/Россия/ЦФО
        Record(category='ad level_kind=1', count=2, fielddate='2019-01-01', region_tree='\t10000\t225\t999999000\t'),  # Земля/Россия/0+
        Record(category='ad level_kind=1', count=1, fielddate='2019-01-01', region_tree='\t10000\t225\t3\t1\t'),       # Земля/Россия/ЦФО/Московская область
        Record(category='ad level_kind=2', count=1, fielddate='2019-01-01', region_tree='\t10000\t'),                  # Земля
        Record(category='ad level_kind=2', count=1, fielddate='2019-01-01', region_tree='\t10000\t225\t'),             # Земля/Россия
        Record(category='ad level_kind=2', count=1, fielddate='2019-01-01', region_tree='\t10000\t225\t3\t'),          # Земля/Россия/ЦФО
        Record(category='ad level_kind=2', count=1, fielddate='2019-01-01', region_tree='\t10000\t225\t3\t1\t'),       # Земля/Россия/ЦФО/Московская область
        Record(category='ad level_kind=2', count=1, fielddate='2019-01-01', region_tree='\t10000\t225\t999999000\t'),  # Земля/Россия/0+
    ])


def test_ad_town_count(statface_client, job, ad_sources):
    job = common.make_count_job(statface_client, job, ['ad'])

    counted_ad_town = []
    job.local_run(
        sources=ad_sources,
        sinks={
            'counted_ad_by_town': local.ListSink(counted_ad_town),
        }
    )
    assert sorted(counted_ad_town) == sorted([
        Record(category='ad town=False', count=1, fielddate='2019-01-01', region_tree='\t10000\t'),                  # Земля
        Record(category='ad town=False', count=1, fielddate='2019-01-01', region_tree='\t10000\t225\t'),             # Земля/Россия
        Record(category='ad town=False', count=1, fielddate='2019-01-01', region_tree='\t10000\t225\t3\t'),          # Земля/Россия/ЦФО
        Record(category='ad town=False', count=1, fielddate='2019-01-01', region_tree='\t10000\t225\t999999000\t'),  # Земля/Россия/0+
        Record(category='ad town=None',  count=1, fielddate='2019-01-01', region_tree='\t10000\t'),                  # Земля
        Record(category='ad town=None',  count=1, fielddate='2019-01-01', region_tree='\t10000\t225\t'),             # Земля/Россия
        Record(category='ad town=None',  count=1, fielddate='2019-01-01', region_tree='\t10000\t225\t3\t'),          # Земля/Россия/ЦФО
        Record(category='ad town=None',  count=1, fielddate='2019-01-01', region_tree='\t10000\t225\t3\t1\t'),       # Земля/Россия/ЦФО/Московская область
        Record(category='ad town=None',  count=1, fielddate='2019-01-01', region_tree='\t10000\t225\t999999000\t'),  # Земля/Россия/0+
        Record(category='ad town=True',  count=1, fielddate='2019-01-01', region_tree='\t10000\t'),                  # Земля
        Record(category='ad town=True',  count=1, fielddate='2019-01-01', region_tree='\t10000\t225\t'),             # Земля/Россия
        Record(category='ad town=True',  count=1, fielddate='2019-01-01', region_tree='\t10000\t225\t3\t'),          # Земля/Россия/ЦФО
        Record(category='ad town=True',  count=1, fielddate='2019-01-01', region_tree='\t10000\t225\t3\t1\t'),       # Земля/Россия/ЦФО/Московская область
        Record(category='ad town=True',  count=1, fielddate='2019-01-01', region_tree='\t10000\t225\t999999000\t'),  # Земля/Россия/0+
    ])


def test_ad_shape_type_count(statface_client, job, ad_sources):
    job = common.make_count_job(statface_client, job, ['ad'])

    counted_ad_shape_type = []
    job.local_run(
        sources=ad_sources,
        sinks={
            'counted_ad_shape_type': local.ListSink(counted_ad_shape_type),
        }
    )
    assert sorted(counted_ad_shape_type) == sorted([
        Record(category='ad all',     count=3, fielddate='2019-01-01', region_tree='\t10000\t'),                  # Земля
        Record(category='ad all',     count=3, fielddate='2019-01-01', region_tree='\t10000\t225\t'),             # Земля/Россия
        Record(category='ad all',     count=3, fielddate='2019-01-01', region_tree='\t10000\t225\t3\t'),          # Земля/Россия/ЦФО
        Record(category='ad all',     count=3, fielddate='2019-01-01', region_tree='\t10000\t225\t999999000\t'),  # Земля/Россия/0+
        Record(category='ad all',     count=2, fielddate='2019-01-01', region_tree='\t10000\t225\t3\t1\t'),       # Земля/Россия/ЦФО/Московская область

        Record(category='ad point',   count=2, fielddate='2019-01-01', region_tree='\t10000\t'),                  # Земля
        Record(category='ad point',   count=2, fielddate='2019-01-01', region_tree='\t10000\t225\t'),             # Земля/Россия
        Record(category='ad point',   count=2, fielddate='2019-01-01', region_tree='\t10000\t225\t3\t'),          # Земля/Россия/ЦФО
        Record(category='ad point',   count=1, fielddate='2019-01-01', region_tree='\t10000\t225\t3\t1\t'),       # Земля/Россия/ЦФО/Московская область
        Record(category='ad point',   count=2, fielddate='2019-01-01', region_tree='\t10000\t225\t999999000\t'),  # Земля/Россия/0+

        Record(category='ad polygon', count=1, fielddate='2019-01-01', region_tree='\t10000\t'),                  # Земля
        Record(category='ad polygon', count=1, fielddate='2019-01-01', region_tree='\t10000\t225\t'),             # Земля/Россия
        Record(category='ad polygon', count=1, fielddate='2019-01-01', region_tree='\t10000\t225\t3\t'),          # Земля/Россия/ЦФО
        Record(category='ad polygon', count=1, fielddate='2019-01-01', region_tree='\t10000\t225\t3\t1\t'),       # Земля/Россия/ЦФО/Московская область
        Record(category='ad polygon', count=1, fielddate='2019-01-01', region_tree='\t10000\t225\t999999000\t'),  # Земля/Россия/0+
    ])


def test_ad_town(statface_client, job, ad_sources):
    job = common.make_count_job(statface_client, job, ['ad'])

    ad_with_town = []
    job.local_run(
        sources=ad_sources,
        sinks={
            'ad_with_town': local.ListSink(ad_with_town),
        }
    )
    assert sorted(ad_with_town) == sorted([
        Record(ad_id=1, count=1, level_kind=1, region_id='1', shape_type='polygon', town=True),
        Record(ad_id=2, count=1, level_kind=1, region_id='3', shape_type='point',   town=False),
        Record(ad_id=3, count=1, level_kind=2, region_id='1', shape_type='point'),
    ])


def test_ad_count(statface_client, job, ad_sources):
    job = common.make_count_job(statface_client, job, ['ad'])

    result_ad = []
    job.local_run(
        sources=ad_sources,
        sinks={
            'result_ad': local.ListSink(result_ad),
        }
    )
    assert sorted(result_ad) == sorted([
        Record(category='ad all',     count=3, fielddate='2019-01-01', region_tree='\t10000\t'),                  # Земля
        Record(category='ad all',     count=3, fielddate='2019-01-01', region_tree='\t10000\t225\t'),             # Земля/Россия
        Record(category='ad all',     count=3, fielddate='2019-01-01', region_tree='\t10000\t225\t3\t'),          # Земля/Россия/ЦФО
        Record(category='ad all',     count=3, fielddate='2019-01-01', region_tree='\t10000\t225\t999999000\t'),  # Земля/Россия/0+
        Record(category='ad all',     count=2, fielddate='2019-01-01', region_tree='\t10000\t225\t3\t1\t'),       # Земля/Россия/ЦФО/Московская область

        Record(category='ad point',   count=2, fielddate='2019-01-01', region_tree='\t10000\t'),                  # Земля
        Record(category='ad point',   count=2, fielddate='2019-01-01', region_tree='\t10000\t225\t'),             # Земля/Россия
        Record(category='ad point',   count=2, fielddate='2019-01-01', region_tree='\t10000\t225\t3\t'),          # Земля/Россия/ЦФО
        Record(category='ad point',   count=1, fielddate='2019-01-01', region_tree='\t10000\t225\t3\t1\t'),       # Земля/Россия/ЦФО/Московская область
        Record(category='ad point',   count=2, fielddate='2019-01-01', region_tree='\t10000\t225\t999999000\t'),  # Земля/Россия/0+

        Record(category='ad polygon', count=1, fielddate='2019-01-01', region_tree='\t10000\t'),                  # Земля
        Record(category='ad polygon', count=1, fielddate='2019-01-01', region_tree='\t10000\t225\t'),             # Земля/Россия
        Record(category='ad polygon', count=1, fielddate='2019-01-01', region_tree='\t10000\t225\t3\t'),          # Земля/Россия/ЦФО
        Record(category='ad polygon', count=1, fielddate='2019-01-01', region_tree='\t10000\t225\t3\t1\t'),       # Земля/Россия/ЦФО/Московская область
        Record(category='ad polygon', count=1, fielddate='2019-01-01', region_tree='\t10000\t225\t999999000\t'),  # Земля/Россия/0+

        Record(category='ad level_kind=1', count=2, fielddate='2019-01-01', region_tree='\t10000\t'),                  # Земля
        Record(category='ad level_kind=1', count=2, fielddate='2019-01-01', region_tree='\t10000\t225\t'),             # Земля/Россия
        Record(category='ad level_kind=1', count=2, fielddate='2019-01-01', region_tree='\t10000\t225\t3\t'),          # Земля/Россия/ЦФО
        Record(category='ad level_kind=1', count=2, fielddate='2019-01-01', region_tree='\t10000\t225\t999999000\t'),  # Земля/Россия/0+
        Record(category='ad level_kind=1', count=1, fielddate='2019-01-01', region_tree='\t10000\t225\t3\t1\t'),       # Земля/Россия/ЦФО/Московская область
        Record(category='ad level_kind=2', count=1, fielddate='2019-01-01', region_tree='\t10000\t'),                  # Земля
        Record(category='ad level_kind=2', count=1, fielddate='2019-01-01', region_tree='\t10000\t225\t'),             # Земля/Россия
        Record(category='ad level_kind=2', count=1, fielddate='2019-01-01', region_tree='\t10000\t225\t3\t'),          # Земля/Россия/ЦФО
        Record(category='ad level_kind=2', count=1, fielddate='2019-01-01', region_tree='\t10000\t225\t3\t1\t'),       # Земля/Россия/ЦФО/Московская область
        Record(category='ad level_kind=2', count=1, fielddate='2019-01-01', region_tree='\t10000\t225\t999999000\t'),  # Земля/Россия/0+

        Record(category='ad town=False', count=1, fielddate='2019-01-01', region_tree='\t10000\t'),                  # Земля
        Record(category='ad town=False', count=1, fielddate='2019-01-01', region_tree='\t10000\t225\t'),             # Земля/Россия
        Record(category='ad town=False', count=1, fielddate='2019-01-01', region_tree='\t10000\t225\t3\t'),          # Земля/Россия/ЦФО
        Record(category='ad town=False', count=1, fielddate='2019-01-01', region_tree='\t10000\t225\t999999000\t'),  # Земля/Россия/0+
        Record(category='ad town=None',  count=1, fielddate='2019-01-01', region_tree='\t10000\t'),                  # Земля
        Record(category='ad town=None',  count=1, fielddate='2019-01-01', region_tree='\t10000\t225\t'),             # Земля/Россия
        Record(category='ad town=None',  count=1, fielddate='2019-01-01', region_tree='\t10000\t225\t3\t'),          # Земля/Россия/ЦФО
        Record(category='ad town=None',  count=1, fielddate='2019-01-01', region_tree='\t10000\t225\t3\t1\t'),       # Земля/Россия/ЦФО/Московская область
        Record(category='ad town=None',  count=1, fielddate='2019-01-01', region_tree='\t10000\t225\t999999000\t'),  # Земля/Россия/0+
        Record(category='ad town=True',  count=1, fielddate='2019-01-01', region_tree='\t10000\t'),                  # Земля
        Record(category='ad town=True',  count=1, fielddate='2019-01-01', region_tree='\t10000\t225\t'),             # Земля/Россия
        Record(category='ad town=True',  count=1, fielddate='2019-01-01', region_tree='\t10000\t225\t3\t'),          # Земля/Россия/ЦФО
        Record(category='ad town=True',  count=1, fielddate='2019-01-01', region_tree='\t10000\t225\t3\t1\t'),       # Земля/Россия/ЦФО/Московская область
        Record(category='ad town=True',  count=1, fielddate='2019-01-01', region_tree='\t10000\t225\t999999000\t'),  # Земля/Россия/0+
    ])
