import common

from nile.api.v1 import (
    local,
    Record
)
import pytest


RD_SHAPE1 = (  # LineString
    '0102000020E6100000090000004C027B1C2B2F44404F4B20E3990E4C407C'
    '74F23B302F4440FDCC0F12980E4C40E3F96FD2322F4440420F57FC960E4C4004597FE'
    '6542F4440DE745E72820E4C405799089F562F4440DCAB1F33810E4C406F116479612F'
    '44402A6213C4790E4C407BF7F2FB712F444074C116446F0E4C4092678FFE742F44403'
    'C47211F6D0E4C40EFF76D83782F4440C341A5BE6A0E4C40'
)
RD_SHAPE2 = (  # MultiLineString
    '0105000020E6100000020000000102000000060000002D836C84BE3044405A0FF2895'
    'B0F4C40F98E84F7C130444097E71225590F4C40E14667ECCD3044405453CFD8500F4C'
    '405BE5CA1FDA3044407CAA2461480F4C40ED822ED2E5304440AFF3FF42400F4C40A3A'
    '12312F5304440C9FBD827370F4C4001020000000A000000DC1AC7FE28314440A86A94'
    '8BF70E4C40BE4833D9293144402E46D1C1020F4C404955EAC42B3144403CACAF2F0B0'
    'F4C40F2E19E562D314440E644CFEE0F0F4C40DA619A892C314440A7FE2D401C0F4C40'
    'C9C798022C314440DFB8F7361E0F4C40389F98C72A314440646A31C91F0F4C4027E89'
    '9D828314440E91B6B5B210F4C4023BA37240D314440EBB094F5330F4C409C9741B702'
    '314440E2CAB82D3A0F4C40'
)
RD_SHAPE3 = (
    '0102000020E6100000030000003569DE67187A44408000AF55567D4C40D55E23CD087'
    'A44401F44F37F557D4C40BBC849D0BB79444089D57561517D4C40'
)


@pytest.fixture
def rd_sources(major_regions):
    return {
        'rd': local.StreamSource([
            Record(rd_id=1, rd_type=1),
            Record(rd_id=2, rd_type=2),
            Record(rd_id=3, rd_type=2),
        ]),
        'rd_geom': local.StreamSource([
            Record(rd_id=1, shape=RD_SHAPE1),
            Record(rd_id=2, shape=RD_SHAPE2),
            Record(rd_id=3, shape=RD_SHAPE3),
        ]),
        'major_regions': local.StreamSource(major_regions),
    }


def test_rd_region(statface_client, job, rd_sources):
    job = common.make_length_job(statface_client, job, ['rd'])

    rd_with_region = []
    job.local_run(
        sources=rd_sources,
        sinks={'rd_with_region': local.ListSink(rd_with_region)}
    )
    assert sorted(rd_with_region) == sorted([
        Record(rd_id=1, rd_type=1, shape=RD_SHAPE1, region_id='192'),  # Владимир
        Record(rd_id=2, rd_type=2, shape=RD_SHAPE2, region_id='192'),  # Владимир
        Record(rd_id=3, rd_type=2, shape=RD_SHAPE3, region_id='5'),    # unknown region
    ])


def test_rd_region_tree(statface_client, job, rd_sources):
    job = common.make_count_job(statface_client, job, ['rd'])

    rd_region_tree = []
    job.local_run(
        sources=rd_sources,
        sinks={'rd_region_tree': local.ListSink(rd_region_tree)}
    )
    assert sorted(rd_region_tree) == sorted([
        Record(count=1, major_id='10000',     rd_type=1, region_id='192', region_tree='\t10000\t'),                  # Земля
        Record(count=1, major_id='10000',     rd_type=2, region_id='192', region_tree='\t10000\t'),                  # Земля
        Record(count=1, major_id='999999003', rd_type=1, region_id='192', region_tree='\t10000\t225\t999999003\t'),  # Земля/Россия/100K+
        Record(count=1,                       rd_type=2, region_id='5',   region_tree='\t10000\t'),                  # Земля
        Record(count=1, major_id='999999003', rd_type=2, region_id='192', region_tree='\t10000\t225\t999999003\t'),  # Земля/Россия/100K+
    ])


def test_rd_count(statface_client, job, rd_sources):
    job = common.make_count_job(statface_client, job, ['rd'])

    result_rd = []
    job.local_run(
        sources=rd_sources,
        sinks={'result_rd': local.ListSink(result_rd)}
    )

    assert sorted(result_rd) == sorted([
        Record(category='rd rd_type=1', count=1, fielddate='2019-01-01', region_tree='\t10000\t'),                  # Земля
        Record(category='rd rd_type=1', count=1, fielddate='2019-01-01', region_tree='\t10000\t225\t999999003\t'),  # Земля/Россия/100K+
        Record(category='rd rd_type=2', count=2, fielddate='2019-01-01', region_tree='\t10000\t'),                  # Земля
        Record(category='rd rd_type=2', count=1, fielddate='2019-01-01', region_tree='\t10000\t225\t999999003\t'),  # Земля/Россия/100K+
        Record(category='rd all',       count=3, fielddate='2019-01-01', region_tree='\t10000\t'),                  # Земля
        Record(category='rd all',       count=2, fielddate='2019-01-01', region_tree='\t10000\t225\t999999003\t'),  # Земля/Россия/100K+
    ])


def test_rd_length(statface_client, job, rd_sources):
    job = common.make_length_job(statface_client, job, ['rd'])

    result_rd = []
    job.local_run(
        sources=rd_sources,
        sinks={'result_rd': local.ListSink(result_rd)}
    )

    assert sorted(result_rd) == sorted([
        Record(category='rd rd_type=1', length=218.00787827236786, fielddate='2019-01-01', region_tree='\t10000\t'),                  # Земля
        Record(category='rd rd_type=1', length=218.00787827236786, fielddate='2019-01-01', region_tree='\t10000\t225\t999999003\t'),  # Земля/Россия/100K+
        Record(category='rd rd_type=2', length=589.5196545591344,  fielddate='2019-01-01', region_tree='\t10000\t'),                  # Земля
        Record(category='rd rd_type=2', length=416.87679676023436, fielddate='2019-01-01', region_tree='\t10000\t225\t999999003\t'),  # Земля/Россия/100K+
        Record(category='rd all',       length=807.5275328315023,  fielddate='2019-01-01', region_tree='\t10000\t'),                  # Земля
        Record(category='rd all',       length=634.8846750326022,  fielddate='2019-01-01', region_tree='\t10000\t225\t999999003\t'),  # Земля/Россия/100K+
    ])
