from maps.wikimap.stat.libs.common.lib import geobase_region
from yandex.maps import geolib3

from nile.api.v1 import (
    extractors as ne,
    stream as ns
)
from qb2.api.v1 import (
    extractors as qe
)

import codecs
import typing as tp

EARTH_REGION_NAME: str = 'Земля'


def point_region(shape: str) -> tp.Optional[str]:
    if not shape:
        return None
    point = geolib3.Point2.from_EWKB(codecs.decode(shape, 'hex'))
    return str(geobase_region.geobase_region_id(point.lon, point.lat))


def add_point_geom(
    stream: ns.Stream,
    ft_center: ns.Stream,
    node: ns.Stream,
    geom_field_name: str
) -> ns.Stream:
    '''
    Input stream:
    | ft_id | ... |
    |-------+-----|
    |   ... | ... |

    ft_center:
    | ft_id | node_id |
    |-------+---------|
    |   ... |     ... |

    node:
    | node_id | shape |
    |---------+-------|
    |     ... |   ... |

    Result stream:
    | ft_id | {geom_field_name} | ... |
    |-------+-------------------+-----|
    |   ... |               ... | ... |
    '''
    return stream.join(
        ft_center,
        by='ft_id',
        type='inner',
        assume_unique=True,
        allow_undefined_keys=False,
        assume_defined=True
    ).join(
        node,
        by='node_id',
        type='inner',
        assume_unique=True,
        allow_undefined_keys=False,
        assume_defined=True
    ).project(
        ne.all(),
        qe.copy(geom_field_name, 'shape')
    )


def add_region_id(
    stream: ns.Stream,
    geom_field_name: str
) -> ns.Stream:
    '''
    Input stream:
    | {geom_field_name} | ... |
    |-------------------+-----|
    |               ... | ... |

    Result stream:
    | {geom_field_name} | region_id | ... |
    |-------------------+-----------+-----|
    |               ... |       ... | ... |
    '''
    return stream.project(
        ne.all(),
        region_id=ne.custom(point_region, geom_field_name).with_type(str),
        files=geobase_region.FILES,
        memory_limit=geobase_region.GEOBASE_JOB_MEMORY_LIMIT
    )


def add_region_name(
    stream: ns.Stream,
    major_regions: ns.Stream
) -> ns.Stream:
    '''
    Input stream:
    | region_id | ... |
    |-----------+-----|
    |       ... | ... |

    major_regions:
    | region_id | region_name | ... |
    |-----------+-------------+-----|
    |       ... |         ... | ... |

    Result stream:
    | region_id | region_name | ... |
    |-----------+-------------+-----|
    |       ... |         ... | ... |
    '''
    return stream.join(
        major_regions,
        by='region_id',
        type='left',
        assume_small_right=True,
        allow_undefined_keys=False,
        assume_defined=True,
        memory_limit=8 * 1024
    ).project(
        ne.all(),
        region_name=ne.custom(
            lambda rn: rn or EARTH_REGION_NAME, 'region_name'
        ).with_type(str)
    )
