import pytest

import maps.wikimap.stat.kpi.valuable_edits.moderated_edits.lib.moderated_edits as me
import maps.wikimap.stat.libs.geobase_regions as gb

from nile.api.v1 import (
    local,
    clusters,
    Record
)


gb.init_geodata_from_resource()
me.GEOBASE_FILES = []


@pytest.fixture
def cluster():
    return clusters.MockCluster()


def make_job(cluster):
    job = cluster.job()

    job = me.make_job(
        job,
        date='2020-10-19',
        logs_path='//logs',
        analytics_path='//analytics',
        results_path='//results'
    )

    return job


def run_job(job, edits_log, moderation_log, users_dump_log):
    result = []

    job.local_run(
        sources={
            "edits_log": local.StreamSource(edits_log),
            "moderation_log": local.StreamSource(moderation_log),
            "users_dump_log": local.StreamSource(users_dump_log)
        },
        sinks={
            "result": local.ListSink(result)
        }
    )

    return result


def test_takes_created_edits(cluster):
    result = run_job(
        make_job(cluster),
        edits_log=[
            Record(
                action='object-modified',
                branch_id='0',
                business_rubric_id='184107202',
                commit_id='100',
                fielddate='2019-03-19',
                ft_type_id='165',
                geom='[[40.9039767299,57.7339356755],[40.9039767479,57.7339356851]]',
                iso_eventtime='2019-03-19 03:00:00',
                object_category='poi_edu',
                object_id='2',
                puid='3',
                source_uri='prt://maps_um_back@um-back01h.maps.yandex.ru/var/lib/yandex/maps/wiki/stat/edits-log.tskv',
            )
        ],
        moderation_log=[
            Record(
                event_type='',
                geom='',
                puid='103',
                revert_reason='',
                resolution='',
                source_uri='',
                task_action='created',
                task_id='4',
                iso_eventtime='2019-03-19 03:00:00',
                object_id='2',
                commit_id='100',
            )
        ],
        users_dump_log=[
            Record(puid='3', moderation_status='moderator')
        ]
    )

    assert result == [
        Record(
            action='object-modified',
            branch_id='0',
            business_rubric_id='184107202',
            commit_id='100',
            fielddate='2019-03-19',
            ft_type_id='165',
            geom='[[40.9039767299,57.7339356755],[40.9039767479,57.7339356851]]',
            iso_eventtime='2019-03-19 03:00:00',
            moderation_task_id='4',
            moderation_status='moderator',
            object_category='poi_edu',
            object_id='2',
            puid='3',
            region_id='7',
            source_uri='prt://maps_um_back@um-back01h.maps.yandex.ru/var/lib/yandex/maps/wiki/stat/edits-log.tskv'
        )
    ]


@pytest.mark.parametrize(
    "task_action,resolution",
    [
        ('resolved', 'accept'),
        ('resolved', 'edit'),
        ('closed', 'approve'),
        ('closed', 'edit')
    ]
)
def test_takes_moderated_in_one_action_edits(cluster, task_action, resolution):
    if task_action == 'resolved':
        moderator_puid = '103'
        action_specific_fields = {
            'moderation_resolved_puid': '103',
            'moderation_resolved_status': 'yandex-moderator',
            'moderation_resolved_resolution': resolution,
            'moderation_resolved_time': '2019-03-25 14:00:00',
        }
    if task_action == 'closed':
        moderator_puid = '104'
        action_specific_fields = {
            'moderation_closed_puid': '104',
            'moderation_closed_status': 'cartographer',
            'moderation_closed_resolution': resolution,
            'moderation_closed_time': '2019-03-25 14:00:00',
            'moderation_revert_reason': ''
        }

    result = run_job(
        make_job(cluster),
        edits_log=[
            Record(
                action='object-modified',
                branch_id='0',
                business_rubric_id='184107202',
                commit_id='1',
                fielddate='2019-03-19',
                ft_type_id='165',
                geom='[[40.9039767299,57.7339356755],[40.9039767479,57.7339356851]]',
                iso_eventtime='2019-03-19 03:00:00',
                object_category='poi_edu',
                object_id='2',
                puid='3',
                source_uri='prt://maps_um_back@um-back01h.maps.yandex.ru/var/lib/yandex/maps/wiki/stat/edits-log.tskv',
            )
        ],
        moderation_log=[
            Record(
                event_type='',
                geom='',
                puid='3',
                revert_reason='',
                resolution='',
                source_uri='',
                task_action='created',
                task_id='4',
                iso_eventtime='2019-03-25 13:00:00',
                object_id='2',
                commit_id='1',
            ),
            Record(
                event_type='',
                geom='',
                puid=moderator_puid,
                revert_reason='',
                resolution=resolution,
                source_uri='',
                task_action=task_action,
                task_id='4',
                iso_eventtime='2019-03-25 14:00:00',
                object_id='3',
                commit_id='1',
            )
        ],
        users_dump_log=[
            Record(puid='3', moderation_status='moderator'),
            Record(puid='103', moderation_status='yandex-moderator'),
            Record(puid='104', moderation_status='cartographer')
        ]
    )

    assert result == [
        Record(
            action='object-modified',
            branch_id='0',
            business_rubric_id='184107202',
            commit_id='1',
            fielddate='2019-03-19',
            ft_type_id='165',
            geom='[[40.9039767299,57.7339356755],[40.9039767479,57.7339356851]]',
            iso_eventtime='2019-03-19 03:00:00',
            moderation_task_id='4',
            moderation_status='moderator',
            object_category='poi_edu',
            object_id='2',
            puid='3',
            region_id='7',
            source_uri='prt://maps_um_back@um-back01h.maps.yandex.ru/var/lib/yandex/maps/wiki/stat/edits-log.tskv',
            **action_specific_fields
        )
    ]


@pytest.mark.parametrize(
    'resolved_resolution,closed_resolution',
    [
        ('edit', 'approve'),
        ('edit', 'edit'),
        ('revert', 'revert')
    ]
)
def test_takes_resolved_and_closed_edits(cluster, resolved_resolution, closed_resolution):
    result = run_job(
        make_job(cluster),
        edits_log=[
            Record(
                action='object-modified',
                branch_id='0',
                business_rubric_id='184107202',
                commit_id='1',
                fielddate='2019-03-19',
                ft_type_id='165',
                geom='[[40.9039767299,57.7339356755],[40.9039767479,57.7339356851]]',
                iso_eventtime='2019-03-19 03:00:00',
                object_category='poi_edu',
                object_id='2',
                puid='3',
                source_uri='prt://maps_um_back@um-back01h.maps.yandex.ru/var/lib/yandex/maps/wiki/stat/edits-log.tskv',
            )
        ],
        moderation_log=[
            Record(
                event_type='',
                geom='',
                puid='3',
                revert_reason='',
                resolution='',
                source_uri='',
                task_action='created',
                task_id='4',
                time='2019-03-25 12:00:00',
                object_id='2',
                commit_id='1',
            ),
            Record(
                event_type='',
                geom='',
                puid='103',
                revert_reason='',
                resolution=resolved_resolution,
                source_uri='',
                task_action='resolved',
                task_id='4',
                iso_eventtime='2019-03-25 13:00:00',
                object_id='2',
                commit_id='1',
            ),
            Record(
                event_type='',
                geom='',
                puid='104',
                revert_reason='',
                resolution=closed_resolution,
                source_uri='',
                task_action='closed',
                task_id='4',
                iso_eventtime='2019-03-25 14:00:00',
                object_id='2',
                commit_id='1'
            )
        ],
        users_dump_log=[
            Record(puid='3', moderation_status='moderator'),
            Record(puid='103', moderation_status='yandex-moderator'),
            Record(puid='104', moderation_status='cartographer')
        ]
    )

    assert result == [
        Record(
            action='object-modified',
            branch_id='0',
            business_rubric_id='184107202',
            commit_id='1',
            fielddate='2019-03-19',
            ft_type_id='165',
            geom='[[40.9039767299,57.7339356755],[40.9039767479,57.7339356851]]',
            iso_eventtime='2019-03-19 03:00:00',
            moderation_task_id='4',
            moderation_status='moderator',
            moderation_resolved_puid='103',
            moderation_resolved_status='yandex-moderator',
            moderation_resolved_resolution=resolved_resolution,
            moderation_resolved_time='2019-03-25 13:00:00',
            moderation_closed_puid='104',
            moderation_closed_status='cartographer',
            moderation_closed_resolution=closed_resolution,
            moderation_closed_time='2019-03-25 14:00:00',
            moderation_revert_reason='',
            object_category='poi_edu',
            object_id='2',
            puid='3',
            region_id='7',
            source_uri='prt://maps_um_back@um-back01h.maps.yandex.ru/var/lib/yandex/maps/wiki/stat/edits-log.tskv'
        )
    ]


def test_takes_multiple_actions_from_single_puid(cluster):
    result = run_job(
        make_job(cluster),
        edits_log=[
            Record(
                action='object-modified',
                branch_id='0',
                business_rubric_id='184107202',
                commit_id='1',
                fielddate='2019-03-19',
                ft_type_id='165',
                geom='[[40.9039767299,57.7339356755],[40.9039767479,57.7339356851]]',
                iso_eventtime='2019-03-19 03:00:00',
                object_category='poi_edu',
                object_id='2',
                puid='3',
                source_uri='prt://maps_um_back@um-back01h.maps.yandex.ru/var/lib/yandex/maps/wiki/stat/edits-log.tskv',
            ),
            Record(
                action='object-created',
                branch_id='0',
                business_rubric_id='',
                commit_id='10',
                fielddate='2019-03-19',
                ft_type_id='',
                geom='[[30.66,61.68],[30.67,61.69]]',
                iso_eventtime='2019-03-19 02:00:00',
                object_category='bld',
                object_id='20',
                puid='3',
                # region_id='10937',
                source_uri='prt://maps_um_back@um-back01h.maps.yandex.ru/var/lib/yandex/maps/wiki/stat/edits-log.tskv',
            )
        ],
        moderation_log=[
            Record(
                event_type='',
                geom='',
                puid='3',
                revert_reason='',
                resolution='',
                source_uri='',
                task_action='created',
                task_id='4',
                time='2019-03-25 12:00:00',
                object_id='2',
                commit_id='1',
            ),
            Record(
                event_type='',
                geom='',
                puid='3',
                revert_reason='',
                resolution='',
                source_uri='',
                task_action='created',
                task_id='40',
                time='2019-03-25 16:00:00',
                object_id='20',
                commit_id='10',
            ),
            Record(
                event_type='',
                geom='',
                puid='103',
                revert_reason='',
                resolution='accept',
                source_uri='',
                task_action='resolved',
                task_id='4',
                iso_eventtime='2019-03-25 13:00:00',
                object_id='2',
                commit_id='1',
            ),
            Record(
                event_type='',
                geom='',
                puid='103',
                revert_reason='',
                resolution='edit',
                source_uri='',
                task_action='resolved',
                task_id='40',
                iso_eventtime='2019-03-25 17:00:00',
                object_id='20',
                commit_id='10',
            ),
            Record(
                event_type='',
                geom='',
                puid='104',
                revert_reason='',
                resolution='edit',
                source_uri='',
                task_action='closed',
                task_id='4',
                iso_eventtime='2019-03-25 14:00:00',
                object_id='2',
                commit_id='1'
            ),
            Record(
                event_type='',
                geom='',
                puid='104',
                revert_reason='',
                resolution='approve',
                source_uri='',
                task_action='closed',
                task_id='40',
                iso_eventtime='2019-03-25 18:00:00',
                object_id='20',
                commit_id='10'
            )
        ],
        users_dump_log=[
            Record(puid='3', moderation_status='moderator'),
            Record(puid='103', moderation_status='yandex-moderator'),
            Record(puid='104', moderation_status='cartographer')
        ]
    )

    assert sorted(result) == [
        Record(
            action='object-created',
            branch_id='0',
            business_rubric_id='',
            commit_id='10',
            fielddate='2019-03-19',
            ft_type_id='',
            geom='[[30.66,61.68],[30.67,61.69]]',
            iso_eventtime='2019-03-19 02:00:00',
            moderation_task_id='40',
            moderation_resolved_puid='103',
            moderation_resolved_status='yandex-moderator',
            moderation_resolved_resolution='edit',
            moderation_resolved_time='2019-03-25 17:00:00',
            moderation_closed_puid='104',
            moderation_closed_resolution='approve',
            moderation_closed_status='cartographer',
            moderation_closed_time='2019-03-25 18:00:00',
            moderation_revert_reason='',
            moderation_status='moderator',
            object_category='bld',
            object_id='20',
            puid='3',
            region_id='10937',
            source_uri='prt://maps_um_back@um-back01h.maps.yandex.ru/var/lib/yandex/maps/wiki/stat/edits-log.tskv'
        ),
        Record(
            action='object-modified',
            branch_id='0',
            business_rubric_id='184107202',
            commit_id='1',
            fielddate='2019-03-19',
            ft_type_id='165',
            geom='[[40.9039767299,57.7339356755],[40.9039767479,57.7339356851]]',
            iso_eventtime='2019-03-19 03:00:00',
            moderation_task_id='4',
            moderation_status='moderator',
            moderation_resolved_puid='103',
            moderation_resolved_status='yandex-moderator',
            moderation_resolved_resolution='accept',
            moderation_resolved_time='2019-03-25 13:00:00',
            moderation_closed_puid='104',
            moderation_closed_status='cartographer',
            moderation_closed_resolution='edit',
            moderation_closed_time='2019-03-25 14:00:00',
            moderation_revert_reason='',
            object_category='poi_edu',
            object_id='2',
            puid='3',
            region_id='7',
            source_uri='prt://maps_um_back@um-back01h.maps.yandex.ru/var/lib/yandex/maps/wiki/stat/edits-log.tskv'
        )
    ]
