from maps.wikimap.stat.kpi.valuable_edits.users_home_regions.lib.users_home_regions import make_users_home_region_job

from nile.api.v1 import (
    Record,
    clusters,
    local
)

from yt.yson import to_yson_type


def job_mock():
    return clusters.MockCluster().job()


def test_should_add_yandexuid_to_puid():
    result = []

    users_dump = [
        Record(puid=b'1'),
        Record(puid=b'2'),
        Record(puid=b'3')
    ]

    crypta_puid_yandexuid_soup = [
        Record(id1Type=b'yandexuid', id2Type=b'puid', id1=b'1001', id2=b'1'),
        Record(id1Type=b'yandexuid', id2Type=b'puid', id1=b'1002', id2=b'2'),
        Record(id1Type=b'yandexuid', id2Type=b'puid', id1=b'1003', id2=b'2'),
    ]

    make_users_home_region_job(
        job_mock(),
        '2020-06-01'
    ).local_run(
        sources={
            'users_dump': local.StreamSource(users_dump),
            'crypta_puid_yandexuid_soup': local.StreamSource(crypta_puid_yandexuid_soup)
        },
        sinks={'nmaps_puids_with_yandexuids': local.ListSink(result)}
    )

    assert sorted(result) == sorted([
        Record(puid=b'1', yandexuid=b'1001'),
        Record(puid=b'2', yandexuid=b'1002'),
        Record(puid=b'2', yandexuid=b'1003'),
    ])


def test_should_drop_puids_without_linked_yandexuid():
    result = []

    users_dump = [
        Record(puid=b'1'),
        Record(puid=b'2'),
    ]

    crypta_puid_yandexuid_soup = [
        Record(id1Type=b'yandexuid', id2Type=b'puid', id1=b'1001', id2=b'1'),
    ]

    make_users_home_region_job(
        job_mock(),
        '2020-06-01'
    ).local_run(
        sources={
            'users_dump': local.StreamSource(users_dump),
            'crypta_puid_yandexuid_soup': local.StreamSource(crypta_puid_yandexuid_soup)
        },
        sinks={'nmaps_puids_with_yandexuids': local.ListSink(result)}
    )

    assert result == [
        Record(puid=b'1', yandexuid=b'1001')
    ]


def test_should_add_region_ids():
    result = []

    nmaps_puids_with_yandexuids = [
        Record(puid=b'1', yandexuid=b'1001'),
        Record(puid=b'2', yandexuid=b'1002'),
        Record(puid=b'2', yandexuid=b'1003'),
    ]

    crypta_yandexuid_with_all_info = [
        Record(id_type=b'yandexuid', id=b'1001', log_region_ids=to_yson_type({b'1': 2})),
        Record(id_type=b'yandexuid', id=b'1002', log_region_ids=to_yson_type({b'2': 2, b'3': 3})),
        Record(id_type=b'yandexuid', id=b'1003', log_region_ids=to_yson_type({b'4': 1, b'3': 7})),
    ]

    make_users_home_region_job(
        job_mock(),
        '2020-06-01'
    ).local_run(
        sources={
            'nmaps_puids_with_yandexuids': local.StreamSource(nmaps_puids_with_yandexuids),
            'crypta_yandexuid_with_all_info': local.StreamSource(crypta_yandexuid_with_all_info)
        },
        sinks={'nmaps_puids_with_region_ids': local.ListSink(result)}
    )

    assert sorted(set(result)) == sorted([
        Record(puid=b'1', region_id=b'1'),
        Record(puid=b'2', region_id=b'2'),
        Record(puid=b'2', region_id=b'3'),
        Record(puid=b'2', region_id=b'4'),
    ])


def test_should_drop_puids_without_linked_regions():
    result = []

    nmaps_puids_with_yandexuids = [
        Record(puid=b'1', yandexuid=b'1001'),
        Record(puid=b'2', yandexuid=b'1002'),
    ]

    crypta_yandexuid_with_all_info = [
        Record(id_type=b'yandexuid', id=b'1002', log_region_ids=None),
    ]

    make_users_home_region_job(
        job_mock(),
        '2020-06-01'
    ).local_run(
        sources={
            'nmaps_puids_with_yandexuids': local.StreamSource(nmaps_puids_with_yandexuids),
            'crypta_yandexuid_with_all_info': local.StreamSource(crypta_yandexuid_with_all_info)
        },
        sinks={'nmaps_puids_with_region_ids': local.ListSink(result)}
    )

    assert result == []


def _sort_major_ids_column(records):
    return [
        Record(record, major_ids=sorted(record['major_ids']))
        for record in records
    ]


def test_should_convert_region_ids_to_major_ids():
    result = []

    nmaps_puids_with_region_ids = [
        Record(puid=b'1', region_id=b'1'),
        Record(puid=b'2', region_id=b'2'),
        Record(puid=b'2', region_id=b'3'),
    ]

    major_regions_map = [
        Record(region_id=b'1', major_id=b'1'),
        Record(region_id=b'1', major_id=b'9'),

        Record(region_id=b'2', major_id=b'2'),
        Record(region_id=b'2', major_id=b'9'),

        Record(region_id=b'3', major_id=b'8'),
        Record(region_id=b'3', major_id=b'2'),
        Record(region_id=b'3', major_id=b'9'),
    ]

    make_users_home_region_job(
        job_mock(),
        '2020-06-01'
    ).local_run(
        sources={
            'nmaps_puids_with_region_ids': local.StreamSource(nmaps_puids_with_region_ids),
            'major_regions_map': local.StreamSource(major_regions_map)
        },
        sinks={'users_home_region': local.ListSink(result)}
    )

    result_with_sorted_major_ids = _sort_major_ids_column(result)

    assert sorted(result_with_sorted_major_ids) == sorted([
        Record(puid=b'1', major_ids=[b'1', b'9']),
        Record(puid=b'2', major_ids=[b'2', b'8', b'9']),
    ])


def test_should_prepare_users_home_region():
    result = []

    users_dump = [
        Record(puid=b'1'),
    ]

    crypta_puid_yandexuid_soup = [
        Record(id1Type=b'yandexuid', id2Type=b'puid', id1=b'1001', id2=b'1'),
    ]

    crypta_yandexuid_with_all_info = [
        Record(id_type=b'yandexuid', id=b'1001', log_region_ids=to_yson_type({b'1': 2, b'2': 4})),
    ]

    major_regions_map = [
        Record(region_id=b'1', major_id=b'1'),
        Record(region_id=b'1', major_id=b'9'),

        Record(region_id=b'2', major_id=b'2'),
        Record(region_id=b'2', major_id=b'1'),
        Record(region_id=b'2', major_id=b'9'),
    ]

    make_users_home_region_job(
        job_mock(),
        '2020-06-01'
    ).local_run(
        sources={
            'users_dump': local.StreamSource(users_dump),
            'crypta_puid_yandexuid_soup': local.StreamSource(crypta_puid_yandexuid_soup),
            'crypta_yandexuid_with_all_info': local.StreamSource(crypta_yandexuid_with_all_info),
            'major_regions_map': local.StreamSource(major_regions_map)
        },
        sinks={'users_home_region': local.ListSink(result)}
    )

    result_with_sorted_major_ids = _sort_major_ids_column(result)

    assert result_with_sorted_major_ids == [
        Record(puid=b'1', major_ids=[b'1', b'2', b'9']),
    ]
