import pytest
from lxml import etree
from maps.wikimap.stat.kpi.valuable_edits.valuable_categories.lib.valuable_categories import extract


def expect_output(xml_content, expected_output):
    xml_tree = etree.fromstring(xml_content)
    pairs_gen = extract(xml_tree)
    assert '\n'.join(sorted(map(str, pairs_gen))) == '\n'.join(sorted(map(str, expected_output)))


def expect_fail(xml_content, exception_class):
    with pytest.raises(exception_class):
        xml_tree = etree.XML(xml_content)
        pairs_gen = extract(xml_tree)
        print('unexpected success with return value:', list(pairs_gen))


def test_cat_is_not_valuable_by_default():
    expect_output(
        '''<?xml version="1.0"?>
        <category-groups xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xsi:noNamespaceSchemaLocation="categories.xsd">
          <category-group id="grp">
            <categories>
              <category id="cat"/>
            </categories>
          </category-group>
        </category-groups>''',
        [
            {'category_group': 'not_valuable', 'object_category': 'cat'}
        ]
    )


def test_group_can_have_label():
    expect_output(
        '''<?xml version="1.0"?>
        <category-groups xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xsi:noNamespaceSchemaLocation="categories.xsd">
          <category-group id="grp">
            <label>{{category-groups:grp_label}}</label>
            <categories>
              <category id="cat"/>
            </categories>
          </category-group>
        </category-groups>''',
        [
            {'category_group': 'not_valuable', 'object_category': 'cat'}
        ]
    )


def test_define_cat_as_valuable():
    expect_output(
        '''<?xml version="1.0"?>
        <category-groups xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xsi:noNamespaceSchemaLocation="categories.xsd">
          <category-group id="grp">
            <categories>
              <category id="vcat" ugc_valuable="true"/>
            </categories>
         </category-group>
        </category-groups>''',
        [
            {'category_group': 'grp', 'object_category': 'vcat'},
            {'category_group': 'all_valuable', 'object_category': 'vcat'}
        ]
    )


def test_define_grp_as_valuable():
    expect_output(
        '''<?xml version="1.0"?>
        <category-groups xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xsi:noNamespaceSchemaLocation="categories.xsd">
          <category-group id="grp" ugc_valuable="true">
            <categories>
              <category id="vcat"/>
            </categories>
         </category-group>
        </category-groups>''',
        [
            {'category_group': 'grp', 'object_category': 'vcat'},
            {'category_group': 'all_valuable', 'object_category': 'vcat'}
        ]
    )


def test_explicitly_define_grp_as_not_valuable():
    expect_output(
        '''<?xml version="1.0"?>
        <category-groups xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xsi:noNamespaceSchemaLocation="categories.xsd">
          <category-group id="nvgrp" ugc_valuable="false">
            <categories>
              <category id="cat"/>
            </categories>
          </category-group>
        </category-groups>''',
        [
            {'category_group': 'not_valuable', 'object_category': 'cat'}
        ]
    )


def test_define_cat_as_not_valuable_in_valuable_grp():
    expect_output(
        '''<?xml version="1.0"?>
        <category-groups xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xsi:noNamespaceSchemaLocation="categories.xsd">
          <category-group id="vgrp" ugc_valuable="true">
            <categories>
              <category id="nvcat" ugc_valuable="false"/>
            </categories>
          </category-group>
        </category-groups>''',
        [
            {'category_group': 'not_valuable', 'object_category': 'nvcat'}
        ]
    )


def test_define_cat_as_valuable_in_grp_explcitly_defined_as_not_valuable():
    expect_output(
        '''<?xml version="1.0"?>
        <category-groups xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xsi:noNamespaceSchemaLocation="categories.xsd">
          <category-group id="nvgrp" ugc_valuable="false">
            <categories>
              <category id="vcat" ugc_valuable="true"/>
            </categories>
          </category-group>
        </category-groups>''',
        [
            {'category_group': 'nvgrp', 'object_category': 'vcat'},
            {'category_group': 'all_valuable', 'object_category': 'vcat'}
        ]
    )


def test_define_multiple_cats_in_single_grp():
    expect_output(
        '''<?xml version="1.0"?>
        <category-groups xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xsi:noNamespaceSchemaLocation="categories.xsd">
          <category-group id="grp">
            <categories>
              <category id="cat1" ugc_valuable="true"/>
              <category id="cat2"/>
            </categories>
          </category-group>
        </category-groups>''',
        [
            {'category_group': 'grp', 'object_category': 'cat1'},
            {'category_group': 'all_valuable', 'object_category': 'cat1'},
            {'category_group': 'not_valuable', 'object_category': 'cat2'}
        ]
    )


def test_define_multiple_grps():
    expect_output(
        '''<?xml version="1.0"?>
        <category-groups xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xsi:noNamespaceSchemaLocation="categories.xsd">
          <category-group id="vgrp" ugc_valuable="true">
            <categories>
              <category id="cat1"/>
            </categories>
          </category-group>
          <category-group id="grp">
            <categories>
              <category id="cat2"/>
            </categories>
          </category-group>
        </category-groups>''',
        [
            {'category_group': 'vgrp', 'object_category': 'cat1'},
            {'category_group': 'all_valuable', 'object_category': 'cat1'},
            {'category_group': 'not_valuable', 'object_category': 'cat2'}
        ]
    )


def test_fail_on_dupl_grp_ids():
    expect_fail(
        '''<?xml version="1.0"?>
        <category-groups xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xsi:noNamespaceSchemaLocation="categories.xsd">
          <category-group id="dgrp">
            <categories>
              <category id="cat1"/>
            </categories>
          </category-group>
          <category-group id="dgrp">
            <categories>
              <category id="cat2"/>
            </categories>
          </category-group>
        </category-groups>''',
        RuntimeError
    )


def test_fail_on_dupl_cat_ids():
    expect_fail(
        '''<?xml version="1.0"?>
        <category-groups xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xsi:noNamespaceSchemaLocation="categories.xsd">
          <category-group id="grp">
            <categories>
              <category id="dcat"/>
              <category id="dcat" ugc_valuable="true"/>
            </categories>
          </category-group>
        </category-groups>''',
        RuntimeError
    )
