from nile.api.v1 import extractors

import re


EDITS_DOMAIN = b'edits'
MODERATION_DOMAIN = b'moderation'
FEEDBACK_DOMAIN = b'feedback'
TRACKER_DOMAIN = b'tracker'

_ASSESSMENT = b'assessment'
_REMOVE_ISSUE_NUMBER_PATTERN = b'-\\d+$'

# NMAPS-14603
_SMM_COMPONENT = b'60367953138a705137be0215'
_EXCLUDED_COMPONENTS = frozenset({_SMM_COMPONENT})


def get_commit_task_id(entity_domain, category_id):
    return b'/'.join((_ASSESSMENT, entity_domain, category_id))


def get_feedback_task_id(entity_domain, feedback_source, feedback_type):
    return b'/'.join((_ASSESSMENT, entity_domain, feedback_source, feedback_type))


def get_tracker_task_id(entity_id, component):
    queue = re.sub(_REMOVE_ISSUE_NUMBER_PATTERN, b'', entity_id)
    return b'/'.join((_ASSESSMENT, TRACKER_DOMAIN, queue.lower(), component))


def add_tracker_task_id(table, issues_table, components_table):
    '''
    table:
    | entity_id (bytes) | other_log_columns... |
    |-------------------+----------------------|
    | issues_table.key  | ...                  |

    issues_table:
    | key | components (array of length 1) | ... |
    |-----+--------------------------------+-----|
    | ... | ...                            | ... |
    Note. AssertionException is raised if number of components is not equal 1.

    components_table:
    | id  | name | ... |
    |-----+------+-----|
    | ... | ...  | ... |

    result:
    | entity_id | task_id | other_log_columns... |
    |-----------+---------+----------------------|
    | ...       | ...     | ...                  |
    '''
    def flatten_and_check_components(records):
        for record in records:
            components = record.get('components')
            assert components is not None, f'Components are not specified in {record}.'

            # NMAPS-14603. A fast fix for issue with two components in GEOCONTENTFB queue.
            # It is okay if the only component is SMM, so remove it only if there
            # are more than one components in the ticket.
            if len(components) > 1:
                components = list(set(components) - _EXCLUDED_COMPONENTS)

            assert len(components) == 1, f'Exactly one component must be specified {record}.'
            yield record.transform('components', component_id=components[0])

    issues_table = issues_table.project(
        'components',
        entity_id='key'
    )
    components_table = components_table.project(
        component_id='id',
        component='name'
    )

    return table.join(
        issues_table,
        by='entity_id',
        type='left',
        assume_unique_right=True
    ).map(
        flatten_and_check_components
    ).join(
        components_table,
        by='component_id',
        type='left',
        assume_unique_right=True
    ).project(
        extractors.all(exclude=['component_id', 'component']),
        task_id=extractors.custom(get_tracker_task_id, 'entity_id', 'component')
    )
