from maps.wikimap.stat.libs.assessment.lib import assessment
from maps.wikimap.stat.libs import nile_ut

from nile.api.v1 import Record
from nile.drivers.common.progress import CommandFailedError
from yt.yson import to_yson_type

import pytest


def test_should_get_commit_task_id():
    assert assessment.get_commit_task_id(b'domain', b'category') == b'assessment/domain/category'


def test_should_get_feedback_task_id():
    assert assessment.get_feedback_task_id(b'domain', b'source', b'type') == b'assessment/domain/source/type'


def test_should_get_tracker_task_id():
    assert assessment.get_tracker_task_id(b'QUEUE-42', b'component') == b'assessment/tracker/queue/component'


def test_should_add_tracker_task_id():
    result = nile_ut.yt_run(
        assessment.add_tracker_task_id,
        table=nile_ut.Table([
            Record(entity_id=b'queue-1',         table_other_column=1),
            Record(entity_id=b'another-queue-3', table_other_column=3),
        ]),
        issues_table=nile_ut.Table([
            Record(key=b'queue-1',         components=to_yson_type([b'component id 1']), issues_other_column=1),
            Record(key=b'queue-2',         components=to_yson_type([b'component id 2']), issues_other_column=2),
            Record(key=b'another-queue-3', components=to_yson_type([b'component id 3']), issues_other_column=3),
        ]),
        components_table=nile_ut.Table([
            Record(id=b'component id 1', name=b'component 1', components_other_column=1),
            Record(id=b'component id 2', name=b'component 2', components_other_column=2),
            Record(id=b'component id 3', name=b'component 3', components_other_column=3),
        ])
    )

    assert sorted([
        Record(entity_id=b'queue-1',         task_id=b'assessment/tracker/queue/component 1',         table_other_column=1),
        Record(entity_id=b'another-queue-3', task_id=b'assessment/tracker/another-queue/component 3', table_other_column=3),
    ]) == sorted(result)


def test_should_fail_adding_tracker_task_id_for_wrong_number_of_components():
    with pytest.raises(CommandFailedError, match='Components are not specified'):
        nile_ut.yt_run(
            assessment.add_tracker_task_id,
            table=nile_ut.Table([
                Record(entity_id=b'queue-1'),
            ]),
            issues_table=nile_ut.Table([
                Record(key=b'queue-1'),
            ]),
            components_table=nile_ut.Table([])
        )

    with pytest.raises(CommandFailedError, match='Exactly one component must be specified'):
        nile_ut.yt_run(
            assessment.add_tracker_task_id,
            table=nile_ut.Table([
                Record(entity_id=b'queue-1'),
            ]),
            issues_table=nile_ut.Table([
                Record(key=b'queue-1', components=to_yson_type([])),
            ]),
            components_table=nile_ut.Table([])
        )

    with pytest.raises(CommandFailedError, match='Exactly one component must be specified'):
        nile_ut.yt_run(
            assessment.add_tracker_task_id,
            table=nile_ut.Table([
                Record(entity_id=b'queue-1'),
            ]),
            issues_table=nile_ut.Table([
                Record(key=b'queue-1', components=to_yson_type(['component 1', 'component 2'])),
            ]),
            components_table=nile_ut.Table([])
        )


def test_nmaps_14603_should_skip_smm_component_if_there_is_another_one():
    result = nile_ut.yt_run(
        assessment.add_tracker_task_id,
        table=nile_ut.Table([
            Record(entity_id=b'queue-42'),
        ]),
        issues_table=nile_ut.Table([
            Record(key=b'queue-42', components=to_yson_type([assessment._SMM_COMPONENT, b'other component'])),
        ]),
        components_table=nile_ut.Table([
            Record(id=assessment._SMM_COMPONENT, name=b'SMM'),
            Record(id=b'other component',        name=b'other component'),
        ])
    )

    assert sorted([
        Record(entity_id=b'queue-42', task_id=b'assessment/tracker/queue/other component'),
    ]) == sorted(result)


def test_nmaps_14603_should_leave_smm_component_if_there_are_no_more_components():
    result = nile_ut.yt_run(
        assessment.add_tracker_task_id,
        table=nile_ut.Table([
            Record(entity_id=b'queue-42'),
        ]),
        issues_table=nile_ut.Table([
            Record(key=b'queue-42', components=to_yson_type([assessment._SMM_COMPONENT])),
        ]),
        components_table=nile_ut.Table([
            Record(id=assessment._SMM_COMPONENT, name=b'SMM'),
        ])
    )

    assert sorted([
        Record(entity_id=b'queue-42', task_id=b'assessment/tracker/queue/SMM'),
    ]) == sorted(result)
