import pytest
from freezegun import freeze_time
from maps.wikimap.stat.libs.common.lib import dates as d


def test_none_run_isodate_error():
    with pytest.raises(TypeError, match='Expected bytes or unicode argument'):
        dates = d._get_dates_to_update(None, 3)
        print(dates)


def test_bad_run_isodate_error():
    with pytest.raises(ValueError, match='wrong datetime \'103-10-10\':.+'):
        with freeze_time('2019-09-03'):
            dates = d._get_dates_to_update('103-10-10', 3)
            print(dates)


def test_zero_period_error():
    with pytest.raises(ValueError, match='period 0 is not a positive integer'):
        with freeze_time('2019-09-03'):
            dates = d._get_dates_to_update('2019-09-03', 0)
            print(dates)


def test_negative_period_error():
    with pytest.raises(ValueError, match='period -1 is not a positive integer'):
        with freeze_time('2019-09-03'):
            dates = d._get_dates_to_update('2019-09-03', -1)
            print(dates)


def test_tomorrow_error():
    with pytest.raises(ValueError, match=r'future run_isodate=2019-09-04, according to today=2019-09-03'):
        with freeze_time('2019-09-03'):
            dates = d._get_dates_to_update('2019-09-04', 3)
            print(dates)


def test_today_length_matches_default_period():
    with freeze_time('2019-09-03'):
        dates = d._get_dates_to_update('2019-09-03', d.MODERATION_MAX_DELAY)
        assert len(dates) == d.MODERATION_MAX_DELAY


def test_today_period_1():
    with freeze_time('2019-09-03'):
        dates = d._get_dates_to_update('2019-09-03', 1)
        assert dates == ['2019-09-02']


def test_today():
    with freeze_time('2019-09-03'):
        dates = d._get_dates_to_update('2019-09-03', 3)
        assert dates == ['2019-08-31', '2019-09-01', '2019-09-02']


def test_yesterday():
    with freeze_time('2019-09-03'):
        dates = d._get_dates_to_update('2019-09-02', 3)
        assert dates == ['2019-08-30']


def test_old_run_isodate():
    with freeze_time('2019-09-03'):
        dates = d._get_dates_to_update('2018-08-28', 3)
        assert dates == ['2018-08-25']


def test_period_crosses_new_year():
    with freeze_time('2019-09-03'):
        dates = d._get_dates_to_update('2019-01-02', 3)
        assert dates == ['2018-12-30']


def test_input_timestamp_underflow():
    with freeze_time('2019-09-03'):
        dates = d._get_dates_to_update('1969-12-31', 3)
        assert dates == ['1969-12-28']


def test_output_timestamp_underflow():
    with freeze_time('2019-09-03'):
        dates = d._get_dates_to_update('1970-01-02', 3)
        assert dates == ['1969-12-30']


def test_future_dates():
    with freeze_time('2039-02-03'):
        dates = d._get_dates_to_update('2039-02-02', 3)
        assert dates == ['2039-01-30']


def test_should_merge_multiple_dates():
    with freeze_time('2019-09-03'):
        dates = d.dates_to_update(['2019-09-01', '2019-09-02'], 3)
        assert dates == ['2019-08-29', '2019-08-30']
