from .geodata import lookup

from yandex.maps.geolib3 import (
    mercator_to_geo,
    BoundingBox,
    Point2,
    Polygon2,
)

import ast
import binascii
import logging


EARTH_REGION_ID = 10000
ATLANTIS_REGION_ID = -1


def geo_coords_to_region_id(lat, lon):
    '''
    Get `region_id` by `lat, lon`. Returns `ATLANTIS_REGION_ID` when unable to determine region.
    '''
    region_id = lookup().get_region_id_by_location(lat, lon)
    return region_id if region_id > 0 else ATLANTIS_REGION_ID


def mercator_geom_to_region_id(hex_encoded_mercator_wkb):
    '''
    Get `region_id` by bbox center from hex-encoded mercator ewkb geometry.
    Returns `EARTH_REGION_ID` for missing geometry (`None`).
    Returns `ATLANTIS_REGION_ID`, if unable to parse geometry or determine region.
    '''
    if hex_encoded_mercator_wkb is None:
        return EARTH_REGION_ID

    try:
        mercator_geom = Polygon2.from_EWKB(binascii.unhexlify(hex_encoded_mercator_wkb.decode()))
        center = mercator_to_geo(mercator_geom.bounding_box().center())
    except:
        logging.warning("Wrong EWKB: '%s'.", hex_encoded_mercator_wkb.decode())
        return ATLANTIS_REGION_ID

    return geo_coords_to_region_id(center.lat, center.lon)


def lon_lat_position_to_region_id(position):
    '''
    Get `region_id` by `[lon, lat]`.
    Returns `EARTH_REGION_ID` for missing position (`None`).
    Returns `ATLANTIS_REGION_ID`, if unable to parse position or determine region.
    '''
    if position is None:
        return EARTH_REGION_ID

    lon, lat = position
    return geo_coords_to_region_id(lat, lon)


def _ensure_str(value):
    if isinstance(value, bytes):
        return value.decode()

    assert isinstance(value, str), "type " + str(type(value)) + " is not a string"
    return value


def lon_lat_bbox_to_region_id(geo_bbox_string):
    '''
    Get `region_id` by `[[lon1, lat1], [lon2, lat2]]` string.
    Returns `EARTH_REGION_ID` for missing bbox (`None`)
    Returns `ATLANTIS_REGION_ID` when unable to parse bbox-string or determine region.
    '''
    if geo_bbox_string is None:
        return EARTH_REGION_ID

    try:
        lon_lat1, lon_lat2 = ast.literal_eval(_ensure_str(geo_bbox_string))
        center = BoundingBox(Point2(*lon_lat1), Point2(*lon_lat2)).center()
    except:
        logging.warning("Bad geo bbox string: '%s'", geo_bbox_string)
        return ATLANTIS_REGION_ID

    return geo_coords_to_region_id(center.lat, center.lon)
