from maps.wikimap.stat.libs import nile_ut

from nile.api.v1 import Record, filters
from nile.api.v1.job import Job as NileJob
from nile.api.v1.stream import PersistentStream
from qb2.api.v1 import typing
from yt.yson import to_yson_type

import pytest


def take(number, table):
    assert type(table) is PersistentStream
    assert type(number) is int
    return table.take(number)


def concat(job, table1, table2):
    assert type(job) is NileJob
    assert type(table1) is PersistentStream
    assert type(table2) is PersistentStream
    return job.concat(table1, table2)


def split(table, predicate):
    return table.split(predicate)


def yql_utf8_to_string(table, column_name):
    return table.cast(**{column_name: typing.String})


def test_should_not_convert_non_special_types():
    nile_ut.yt_run(take, 42, table=nile_ut.Table([]))


def test_should_convert_special_types_in_args():
    result = nile_ut.yt_run(
        concat,
        nile_ut.Job(),
        nile_ut.Table([Record(a=1)]),
        nile_ut.Table([Record(b=2)]),
    )
    assert sorted(result) == sorted([Record(a=1), Record(b=2)])


def test_should_convert_special_types_in_kwargs():
    result = nile_ut.yt_run(
        concat,
        job=nile_ut.Job(),
        table1=nile_ut.Table([Record(a=1)]),
        table2=nile_ut.Table([Record(b=2)]),
    )
    assert sorted(result) == sorted([Record(a=1), Record(b=2)])


def test_should_set_schema_for_arg_table():
    record = Record(a=to_yson_type({b'n': 42, b's': b'42'}))
    with pytest.raises(ValueError, match='please specify explicit schema'):
        nile_ut.yql_run(take, 1, nile_ut.Table([record]))

    result = nile_ut.yql_run(take, 1, nile_ut.Table([record], schema={'a': typing.Yson}))
    assert result == [record]


def test_should_set_schema_for_kwarg_table():
    record = Record(a=to_yson_type({b'n': 42, b's': b'42'}))
    with pytest.raises(ValueError, match='please specify explicit schema'):
        nile_ut.yql_run(take, 1, table=nile_ut.Table([record]))

    result = nile_ut.yql_run(take, 1, table=nile_ut.Table([record], schema={'a': typing.Yson}))
    assert result == [record]


def test_should_process_several_outputs():
    even, odd = nile_ut.yt_run(
        split,
        table=nile_ut.Table([Record(a=1), Record(a=2), Record(a=3)]),
        predicate=filters.custom(lambda a: bool(a % 2), 'a')
    )
    assert sorted(even) == sorted([Record(a=2)])
    assert sorted(odd) == sorted([Record(a=1), Record(a=3)])


def test_should_run_on_yql():
    result = nile_ut.yql_run(
        yql_utf8_to_string,
        table=nile_ut.Table([Record(a='value')]),
        column_name='a',
    )
    assert result == [Record(a=b'value')]


def test_should_not_run_on_yt():
    with pytest.raises(TypeError):
        nile_ut.yt_run(
            yql_utf8_to_string,
            table=nile_ut.Table([Record(a='value')]),
            column_name='a',
        )


def test_example():
    def swap_columns(table):
        return table.project(a='b', b='a')

    result = nile_ut.yt_run(
        swap_columns,
        table=nile_ut.Table([
            Record(a=1, b=1),
            Record(a=2, b=4),
            Record(a=3, b=9),
        ])
    )
    assert sorted(result) == sorted([
        Record(b=1, a=1),
        Record(b=2, a=4),
        Record(b=3, a=9),
    ])
