from maps.wikimap.stat.libs.nile_utils import ensure_str
from nile.api.v1 import aggregators, extractors, Record, utils, modified_schema
from qb2.api.v1 import typing, filters
import datetime as dt

_UNKNOWN_COMPONENT = 'unknown_component'


def _str_equals_filter(column, value):  # yt/yql interoperability workaround
    return filters.custom(
        lambda column_value: ensure_str(column_value) == value,
        column,
    )


def _str_one_of_filter(column, collection):  # yt/yql interoperability workaround
    return filters.custom(
        lambda column_value: ensure_str(column_value) in collection,
        column,
    )


def filter_by_queues(issues, queues):
    '''
    Drop issues that are not from specified queues.

    issues, result:
    | key | ... |
    |-----|-----|
    | ... | ... |

    '''
    def is_from_queues(key):
        queue, _ = ensure_str(key).split('-', 1)
        return queue in queues

    return issues.filter(
        filters.custom(is_from_queues, 'key')
    )


def _get_all_changes(issue_events_table):
    '''
    issue_events_table:
    | issue | date (ms) | changes (yson)             | ... |
    |-------+-----------+----------------------------+-----|
    | ...   | ...       | [                          | ... |
    |       |           |   {                        |     |
    |       |           |      'field': 'fieldName', |     |
    |       |           |      'newValue': {         |     |
    |       |           |        'value': yson_value |     |
    |       |           |      },                    |     |
    |       |           |      ...                   |     |
    |       |           |   },                       |     |
    |       |           |   ...                      |     |
    |       |           | ]                          |     |

    Result:
    | issue_id | date (ms) | field     | value      |
    |----------+-----------+-----------+------------|
    | ...      | ...       | fieldName | yson_value |
    '''

    @utils.with_hints(
        {
            'issue_id': typing.Unicode,
            'date': typing.Int64,
            'field': typing.Unicode,
            'value': typing.Yson,
        }
    )
    def get_all_changes(records):
        for record in records:
            for change in record.changes:
                yield Record(
                    issue_id=ensure_str(record['issue']),
                    date=record['date'],
                    field=ensure_str(change[b'field']),
                    value=change[b'newValue'][b'value'],
                )

    return issue_events_table.map(get_all_changes)


def _timestamp_to_date(timestamp):
    return dt.date.fromtimestamp(timestamp / 1000).isoformat()


def _get_issues_closed_at(dates, all_changes, statuses_table):
    '''
    all_changes:
    | issue_id | date (ms) | field     | value      |
    |----------+-----------+-----------+------------|
    | ...      | ...       | fieldName | yson_value |

    yson_value for `status` field is {'id': 'status_id'}

    statuses_table:
    | id  | key | ... |
    |-----+-----+-----|
    | ... | ... | ... |

    Result:
    | issue_id | date           |
    |----------+----------------|
    | ...      | <closed_date>  |
    '''

    closed_statuses = statuses_table.filter(
        _str_equals_filter('key', 'closed')
    ).project(
        status_id='id'
    ).label('closed_statuses')

    statuses_changes = all_changes.filter(
        _str_equals_filter('field', 'status')
    ).project(
        'issue_id',
        'date',
        # Assume that status can't change to None, therefore value must be defined.
        status_id=extractors.custom(lambda value: value[b'id'].decode(), 'value').with_type(typing.Unicode)
    ).label('statuses_changes')

    closed_changes = statuses_changes.join(
        closed_statuses,  # Effectively just filtering out all but closed issues.
        by='status_id',
        type='inner'
    ).project(
        'issue_id', 'date'
    ).label('closed_changes')

    first_closed_changes = closed_changes.groupby(
        'issue_id'
    ).aggregate(
        date=aggregators.min('date'),
    ).label('first_closed_changes')

    return first_closed_changes.project(
        'issue_id',
        date=extractors.custom(_timestamp_to_date, 'date').with_type(typing.Unicode),
    ).filter(
        _str_one_of_filter('date', frozenset(dates))
    )


def _get_issues_created_at(dates, issues_table):
    '''
    issues_table:
    | id  | key                   | author | components | created | ... |
    |-----+-----------------------+--------+------------+---------+-----|
    | ... | <queue name>-<number> | ...    | ...        | ...     | ... |

    Result:
    | ticket | staff_uid | components | date             |
    |--------+-----------+------------+------------------|
    | ...    | ...       | ...        | <created>.date() |
    '''

    return issues_table.project(
        'components',
        staff_uid='author',
        ticket='key',
        date=extractors.custom(_timestamp_to_date, 'created').with_type(typing.Unicode),
    ).filter(
        _str_one_of_filter('date', frozenset(dates))
    )


def _get_issues_with_paid_resolutions(issues_closed_at_date, issues_table, resolutions_table):
    '''
    Leaves only information about issues closed with 'fixed' resolutions.

    issues_closed_at_date:
    | issue_id | date |
    |----------+------|
    | ...      | ...  |

    issues_table:
    | id  | key                   | resolution | assignee | components | storyPoints | ... |
    |-----+-----------------------+------------+----------+------------+-------------+-----|
    | ... | <queue name>-<number> | ...        | ...      | ...        | ...         | ... |

    resolutions_table:
    | id  | key                           | ... |
    |-----+-------------------------------+-----|
    | ... | internal name of a resolution | ... |

    Result:
    | staff_uid | components | ticket | story_points | date |
    |-----------+------------+--------+--------------+------|
    | ...       | ...        | ...    | ...          | ...  |
    '''

    paid_resolutions = resolutions_table.filter(
        _str_equals_filter('key', 'fixed')
    ).project(
        resolution_id='id'
    ).label('paid_resolutions')

    return issues_table.project(
        'components',
        staff_uid='assignee',
        issue_id='id',
        resolution_id='resolution',
        ticket='key',
        story_points='storyPoints'
    ).join(
        issues_closed_at_date,  # Effectively just filtering out all but interesting issues.
        by='issue_id',
        type='inner'
    ).join(
        paid_resolutions,  # Effectively just filtering out all but paid resolutions
        by='resolution_id',
        type='inner'
    ).project(
        'staff_uid', 'components', 'ticket', 'story_points', 'date'
    )


def _add_components(issues, components_table):
    '''
    issues:
    | components | ... |
    |------------+-----|
    | ...        | ... |

    components_table:
    | id  | name | ... |
    |-----+------+-----|
    | ... | ...  | ... |

    Result:
    | component      | ... |
    |----------------+-----|
    | component_name | ... |
    '''

    @utils.with_hints(modified_schema(extend={'component_id': typing.String}, exclude=['components']))
    def flat_components(records):
        for record in records:
            if len(record.components) == 1:
                yield record.transform('components', component_id=ensure_str(record.components[0]))
            else:
                yield record.transform('components', component_id=_UNKNOWN_COMPONENT)

    components_table = components_table.project(component_id='id', component_name='name')

    return issues.map(
        flat_components
    ).join(
        components_table,
        by='component_id',
        type='left'
    ).project(
        extractors.all(exclude=['component_name', 'component_id']),
        component=extractors.custom(
            lambda component_name, component_id: component_name if component_name else component_id,
            'component_name', 'component_id'
        ).with_type(typing.Unicode)
    )


def get_closed_issues(
    dates,
    components_table,
    issue_events_table,
    issues_table,
    resolutions_table,
    statuses_table
):
    '''
    Extracts information about Tracker issues for the `dates` from
    the provided tables.

    components_table:
    | id  | name | ... |
    |-----+------+-----|
    | ... | ...  | ... |

    issue_events_table:
    | issue | date (ms) | changes (yson)             | ... |
    |-------+-----------+----------------------------+-----|
    | ...   | ...       | [                          | ... |
    |       |           |   {                        |     |
    |       |           |      'field': 'fieldName', |     |
    |       |           |      'newValue': {         |     |
    |       |           |        'value': yson_value |     |
    |       |           |      },                    |     |
    |       |           |      ...                   |     |
    |       |           |   },                       |     |
    |       |           |   ...                      |     |
    |       |           | ]                          |     |

    issues_table:
    | id  | key                   | resolution | assignee | components | storyPoints | ... |
    |-----+-----------------------+------------+----------+------------+-------------+-----|
    | ... | <queue name>-<number> | ...        | ...      | ...        | ...         | ... |

    resolutions_table:
    | id  | key                           | ... |
    |-----+-------------------------------+-----|
    | ... | internal name of a resolution | ... |

    statuses_table:
    | id  | key                       | ... |
    |-----+---------------------------+-----|
    | ... | internal name of a status | ... |

    Result:
    | staff_uid | component | ticket                | story_points | date          |
    |-----------+-----------+-----------------------+--------------+---------------|
    | ...       | ...       | <queue name>-<number> | ...          | <closed_date> |
    '''

    all_changes = _get_all_changes(issue_events_table)
    issues_closed_at_date = _get_issues_closed_at(dates, all_changes, statuses_table)

    issues_with_paid_resolutions = _get_issues_with_paid_resolutions(
        issues_closed_at_date, issues_table, resolutions_table
    )

    issues_with_components_and_staff_uid = _add_components(
        issues_with_paid_resolutions, components_table
    )

    return issues_with_components_and_staff_uid


def get_created_issues(
    dates,
    components_table,
    issues_table
):
    '''
    Extracts information about created Tracker issues for the `dates` from
    the provided tables.

    components_table:
    | id  | name | ... |
    |-----+------+-----|
    | ... | ...  | ... |

    issues_table:
    | id  | key                   | resolution | assignee | components | storyPoints | created | ... |
    |-----+-----------------------+------------+----------+------------+-------------+---------+-----|
    | ... | <queue name>-<number> | ...        | ...      | ...        | ...         | ...     | ... |

    Result:
    | staff_uid | component | ticket                | story_points | date              |
    |-----------+-----------+-----------------------+--------------+-------------------|
    | ...       | ...       | <queue name>-<number> | ...          | <created>.date()  |
    '''

    issues_created_at_date = _get_issues_created_at(dates, issues_table)

    issues_with_components_and_staff_uid = _add_components(
        issues_created_at_date, components_table
    )

    return issues_with_components_and_staff_uid


def check_issues(records):
    '''
    Raises RuntimeError if any issue contains an incorrect value.

    records, result:
    | staff_uid | component | ticket | story_points | ... |
    |-----------+-----------+--------+--------------+-----|
    | ...       | ...       | ...    | ...          | ... |

    Iterator of Record(staff_uid, component, ticket, story_points)
    '''

    errors = []
    for record in records:
        if record.get('staff_uid') is None:
            errors.append(f'Staff UId is absent in {record}.')
        if record.get('component') is None:
            errors.append(f'Component is absent in {record}.')
        if ensure_str(record.get('component')) == _UNKNOWN_COMPONENT:
            errors.append(f'Unknown component in {record}.')
        if record.get('story_points') is None:
            errors.append(f'Story points are absent in {record}.')

        yield record

    if len(errors) != 0:
        raise RuntimeError('\n'.join(('Issues contains incorrect values:', *errors)))
