from nile.api.v1 import Record


DEFAULT_TREE_ROOT = '\t_total_\t'


def convert_to_utf(field):
    if type(field) == bytes:
        return field.decode('utf-8')
    else:
        return field


def make_role_path(role, user_name, puid):
    role = convert_to_utf(role)
    user_name = convert_to_utf(user_name)
    puid = convert_to_utf(puid)

    path_string = DEFAULT_TREE_ROOT
    yield path_string

    path_string += role.split('.')[0] + '\t'
    yield path_string

    if role.count('.') > 0:
        path_string += 'layer\t'
        yield path_string

        for group in role.split('.', 1)[1].split('.'):
            if group not in ('auto', 'not'):
                yield '{}{}\t'.format(path_string, group)
                yield '{}{}\t{} ({})\t'.format(path_string, group, user_name, puid)
    else:
        path_string += 'all\t'
        yield path_string
        path_string += '{} ({})\t'.format(user_name, puid)
        yield path_string


def add_role_path_mapper(records):
    '''
    Create tree for role_path.

    records
    | puid | fielddate | role |
    |------+-----------+------|
    |  ... |       ... |  ... |

    result
    | puid | fielddate |                                                  role_path |
    |------+-----------+------------------------------------------------------------|
    |  ... |       ... |                                              '\t_total_\t' |
    |  ... |       ... |                                      '\t_total_\texpert\t' |
    |  ... |       ... |                               '\t_total_\texpert\tlayer\t' |
    |  ... |       ... |                   '\t_total_\texpert\tlayer\taddr_group\t' |
    |  ... |       ... | '\t_total_\texpert\tlayer\taddr_group\tuser_name (puid)\t' |
    '''
    for record in records:
        for role_path_element in make_role_path(record.role, record.um_login, record.puid):
            yield Record(
                puid=record.puid,
                fielddate=record.fielddate,
                role_path=role_path_element
            )
