from nile.api.v1 import (
    aggregators as na,
    extractors as ne,
    filters as nf
)
from yt.wrapper.ypath import ypath_join

from . import edit_processors

DEFAULT_NMAP_ACL_ROLES_PATH = '//home/logfeller/logs/nmaps-acl-roles-dump/1d'
DEFAULT_NMAP_USER_PATH = '//home/logfeller/logs/nmaps-users-dump-log/1d'


def make_fielddate(datefield):
    return str(edit_processors.convert_to_utf(datefield)).split(' ')[0]


def make_acl_roles(job, date):
    '''
    Filtered nmaps-acl-roles-dump by role

    result:
    | puid | role |
    |------+------|
    |  ... |  ... |
    '''
    return job.table(
        ypath_join(DEFAULT_NMAP_ACL_ROLES_PATH, date)
    ).filter(
        nf.custom(
            lambda role: edit_processors.convert_to_utf(role).split('.')[0] in ('expert', 'moderator'),
            'role'
        )
    ).project(
        'role',
        puid=ne.custom(
            lambda puid: str(puid)[:-1],
            'puid'
        )
    ).label('acl_roles_raw')


def make_nmaps_user_raw(job, date):
    '''
    result
    | puid | um_login | fielddate |
    |------+----------+-----------|
    |  ... |      ... |       ... |
    '''
    return job.table(
        ypath_join(DEFAULT_NMAP_USER_PATH, date)
    ).project(
        'puid',
        'um_login',
        fielddate=ne.custom(make_fielddate, 'iso_eventtime')
    ).label('nmaps_user_raw')


def add_paths(job, nmaps_acl_roles_raw, nmaps_user_raw):
    '''
    Add trees for event_type, object_type and add user status (moderator, yandex-moderator, cartographer)

    nmaps_acl_roles_raw:
    | puid | role |
    |------+------|
    |  ... |  ... |

    nmaps_user_raw:
    | puid | um_login | fielddate |
    |------+----------+-----------|
    |  ... |      ... |       ... |

    nmaps_acl_roles:
    | puid | fielddate | role |
    |------+-----------+------|
    |  ... |       ... |  ... |

    result:
    | puid | fielddate |  role_path |
    |------+-----------+------------|
    |  ... |       ... |        ... |
    '''
    nmaps_acl_roles = nmaps_user_raw.join(
        nmaps_acl_roles_raw,
        by=('puid'),
        type='inner'
    ).label('user_raw_join_acl_roles')

    return nmaps_acl_roles.map(
        edit_processors.add_role_path_mapper
    ).label('nmaps_acl_roles_add_path')


def prepare_result_counts(stream):
    '''
    Calculating the result of statistics.
    Stream is aggregated by 3 unique values: commit_id, object_id, and puid.

    stream
    | puid | fielddate |  role_path |
    |------+-----------+------------|
    |  ... |       ... |        ... |

    result
    | fielddate | role_path | count_uniq_puid |
    |-----------+-----------+-----------------|
    |       ... |       ... |             ... |
    '''

    return stream.unique(
        'fielddate',
        'puid',
        'role_path'
    ).groupby(
        'fielddate',
        'role_path'
    ).aggregate(
        count_uniq_puid=na.count()
    ).project(
        'fielddate',
        'role_path',
        count_uniq_puid=ne.custom(
            lambda count_uniq_puid: count_uniq_puid or 0,
            'count_uniq_puid'
        )
    ).label('prepare_result_counts')
