from nile.api.v1 import Record

from . import edits_constants


DEFAULT_TREE_ROOT = '\t_total_\t'


def convert_to_utf(field):
    if type(field) == bytes:
        return field.decode('utf-8')
    else:
        return field


def add_event_path_mapper(records):
    '''
    Create tree for event_path.

    records
    | puid | commit_id | fielddate | object_path | event_type |
    |------+-----------+-----------+-------------+------------|
    |  ... |       ... |       ... |         ... |        ... |

    result
    | puid | commit_id | fielddate | object_path |          event_path |
    |------+-----------+-----------+-------------+---------------------|
    |  ... |       ... |       ... |         ... |       '\t_total_\t' |
    |  ... |       ... |       ... |         ... | '\t_total_\tedit\t' |
    |  ... |       ... |       ... |         ... |                 ... |
    '''
    for record in records:
        yield Record(
            puid=record.puid,
            commit_id=record.commit_id,
            fielddate=record.fielddate,
            object_path=record.object_path,
            event_path=DEFAULT_TREE_ROOT
        )
        yield Record(
            puid=record.puid,
            commit_id=record.commit_id,
            fielddate=record.fielddate,
            object_path=record.object_path,
            event_path=DEFAULT_TREE_ROOT + '{}\t'.format(convert_to_utf(record.event_type))
        )


def add_status_path_mapper(records):
    '''
    Create tree for status_path.

    records
    | puid | commit_id | moderation_status | fielddate | object_path | event_path |
    |------+-----------+-------------------+-----------+-------------+------------|
    |  ... |       ... |               ... |       ... |         ... |        ... |

    result
    | puid | commit_id | event_path | fielddate | object_path |                 status_path |
    |------+-----------+------------+-----------+-------------+-----------------------------|
    |  ... |       ... |        ... |       ... |         ... |               '\t_total_\t' |
    |  ... |       ... |        ... |       ... |         ... | '\t_total_\tcartographer\t' |
    |  ... |       ... |        ... |       ... |         ... |    '\t_total_\tmoderator\t' |
    |  ... |       ... |        ... |       ... |         ... |                         ... |
    '''
    for record in records:
        yield Record(
            puid=record.puid,
            commit_id=record.commit_id,
            fielddate=record.fielddate,
            object_path=record.object_path,
            event_path=record.event_path,
            status_path=DEFAULT_TREE_ROOT
        )
        yield Record(
            puid=record.puid,
            commit_id=record.commit_id,
            fielddate=record.fielddate,
            object_path=record.object_path,
            event_path=record.event_path,
            status_path=DEFAULT_TREE_ROOT + '{}\t'.format(convert_to_utf(record.moderation_status))
        )


def add_object_path_mapper(records):
    '''
    Create tree for object_path.
    Take group from dict in edits_constats.DEFINE_CATEGORY_GROUPS_DICT

    records
    | puid | commit_id | fielddate | object_category | event_type |
    |------+-----------+-----------+-----------------+------------|
    |  ... |       ... |       ... |             ... |        ... |

    result
    | puid | commit_id | fielddate | event_type |                    object_path |
    |------+-----------+-----------+------------+--------------------------------|
    |  ... |       ... |       ... |       ... |                   '\t_total_\t' |
    |  ... |       ... |       ... |       ... |         '\t_total_\trd_group\t' |
    |  ... |       ... |       ... |       ... |  '\t_total_\trd_group\trd_jc\t' |
    |  ... |       ... |       ... |       ... |                             ... |
    '''
    for record in records:
        yield Record(
            puid=record.puid,
            commit_id=record.commit_id,
            event_type=record.event_type,
            fielddate=record.fielddate,
            object_path=DEFAULT_TREE_ROOT
        )
        yield Record(
            puid=record.puid,
            commit_id=record.commit_id,
            event_type=record.event_type,
            fielddate=record.fielddate,
            object_path=DEFAULT_TREE_ROOT + '{}\t'.format(
                edits_constants.DEFINE_CATEGORY_GROUPS_DICT.get(convert_to_utf(record.object_category), 'None')
            )
        )
        yield Record(
            puid=record.puid,
            commit_id=record.commit_id,
            event_type=record.event_type,
            fielddate=record.fielddate,
            object_path=DEFAULT_TREE_ROOT + '{}\t{}\t'.format(
                edits_constants.DEFINE_CATEGORY_GROUPS_DICT.get(convert_to_utf(record.object_category), 'None'),
                convert_to_utf(record.object_category)
            )
        )
