from nile.api.v1 import Record

from . import edits_constants


DEFAULT_TREE_ROOT = '\t_total_\t'


def convert_to_utf(field):
    if type(field) == bytes:
        return field.decode('utf-8')
    else:
        return field


def create_user_path(moderation_status, login, puid):
    '''
    Algorithm for create tree for user_path.
    if moderation status in edits_constants.YA_STATUS_LIST - status is 'yandex', else - 'user'
    :return: generator with tree for every user
    '''
    moderation_status = convert_to_utf(moderation_status)
    login = convert_to_utf(login)
    puid = convert_to_utf(puid)

    path_string = DEFAULT_TREE_ROOT
    yield path_string

    path_string += '{}\t'.format('yandex' if moderation_status in edits_constants.YA_STATUS_LIST else 'user')
    yield path_string

    path_string += '{}\t'.format(moderation_status if moderation_status else 'None')
    yield path_string

    path_string += '{} ({})\t'.format(login, puid)
    yield path_string


def add_action_path_mapper(records):
    '''
    Create tree for action_path.

    :param records:
    | puid | commit_id | fielddate | ... | action |
    |------+-----------+-----------+-----+--------|
    |  ... |       ... |       ... | ... |    ... |

    :return:
    | puid | commit_id | fielddate | ... |                    action_path |
    |------+-----------+-----------+-----+--------------------------------|
    |  ... |       ... |       ... | ... |                  '\t_total_\t' |
    |  ... |       ... |       ... | ... | '\t_total_\tcommit-reverted\t' |
    |  ... |       ... |       ... | ... |                            ... |
    '''
    for record in records:
        yield Record(
            puid=record.puid,
            commit_id=record.commit_id,
            object_id=record.object_id,
            task_id=record.task_id,
            region_id=record.region_id,
            resolution=record.resolution,
            unixtime=record.unixtime,
            fielddate=record.fielddate,
            edit_commit_id=record.edit_commit_id,
            edit_object_id=record.edit_object_id,
            moderation_status=record.moderation_status,
            role=record.role,
            object_path=record.object_path,
            user_path=record.user_path,
            action_path=DEFAULT_TREE_ROOT
        )
        yield Record(
            puid=record.puid,
            commit_id=record.commit_id,
            object_id=record.object_id,
            task_id=record.task_id,
            region_id=record.region_id,
            resolution=record.resolution,
            unixtime=record.unixtime,
            fielddate=record.fielddate,
            edit_commit_id=record.edit_commit_id,
            edit_object_id=record.edit_object_id,
            moderation_status=record.moderation_status,
            role=record.role,
            object_path=record.object_path,
            user_path=record.user_path,
            action_path=DEFAULT_TREE_ROOT + '{}\t'.format(convert_to_utf(record.action))
        )


def add_user_path_mapper(records):
    '''
    Create tree for userpath.

    :param records:
    | puid | commit_id | ... | moderation_status |
    |------+-----------+-----+-------------------|
    |  ... |       ... | ... |               ... |

    :return:
    | puid | commit_id | ... | moderation_status |                                              user_path |
    |------+-----------+-----+-------------------+--------------------------------------------------------|
    |  ... |       ... | ... |               ... |                                          '\t_total_\t' |
    |  ... |       ... | ... |               ... |                                  '\t_total_\tyandex\t' |
    |  ... |       ... | ... |               ... |                '\t_total_\tyandex\tyandex-moderator\t' |
    |  ... |       ... | ... |               ... | '\t_total_\tyandex\tyandex-moderator\ttest (123456)\t' |
    |  ... |       ... | ... |               ... |                                                    ... |
    '''
    for record in records:
        for user_string in create_user_path(record.moderation_status, record.um_login, record.puid):
            yield Record(
                puid=record.puid,
                action=record.action,
                commit_id=record.commit_id,
                task_id=record.task_id,
                object_id=record.object_id,
                region_id=record.region_id,
                resolution=record.resolution,
                unixtime=record.unixtime,
                fielddate=record.fielddate,
                edit_commit_id=record.edit_commit_id,
                edit_object_id=record.edit_object_id,
                moderation_status=record.moderation_status,
                role=record.role,
                object_path=record.object_path,
                user_path=user_string
            )


def add_object_path_mapper(records):
    '''
    Create tree for object_path.
    Take group from dict in edits_constats.DEFINE_CATEGORY_GROUPS_DICT

    :param records:
    | puid | commit_id | fielddate | object_category | event_type |
    |------+-----------+-----------+-----------------+------------|
    |  ... |       ... |       ... |             ... |        ... |

    :return:
    | puid | commit_id | fielddate | event_type |                    object_path |
    |------+-----------+-----------+------------+--------------------------------|
    |  ... |       ... |       ... |       ... |                   '\t_total_\t' |
    |  ... |       ... |       ... |       ... |         '\t_total_\trd_group\t' |
    |  ... |       ... |       ... |       ... |  '\t_total_\trd_group\trd_jc\t' |
    |  ... |       ... |       ... |       ... |                             ... |
    '''
    for record in records:
        yield Record(
            puid=record.puid,
            action=record.action,
            commit_id=record.commit_id,
            task_id=record.task_id,
            object_id=record.object_id,
            region_id=record.region_id,
            resolution=record.resolution,
            unixtime=record.unixtime,
            fielddate=record.fielddate,
            edit_commit_id=record.edit_commit_id,
            edit_object_id=record.edit_object_id,
            um_login=record.um_login,
            moderation_status=record.moderation_status,
            role=record.role,
            object_path=DEFAULT_TREE_ROOT
        )
        yield Record(
            puid=record.puid,
            action=record.action,
            commit_id=record.commit_id,
            task_id=record.task_id,
            object_id=record.object_id,
            region_id=record.region_id,
            resolution=record.resolution,
            unixtime=record.unixtime,
            fielddate=record.fielddate,
            edit_commit_id=record.edit_commit_id,
            edit_object_id=record.edit_object_id,
            um_login=record.um_login,
            moderation_status=record.moderation_status,
            role=record.role,
            object_path=DEFAULT_TREE_ROOT + '{}\t'.format(
                edits_constants.DEFINE_CATEGORY_GROUPS_DICT.get(convert_to_utf(record.object_category), 'None')
            )
        )
        yield Record(
            puid=record.puid,
            action=record.action,
            commit_id=record.commit_id,
            task_id=record.task_id,
            object_id=record.object_id,
            region_id=record.region_id,
            resolution=record.resolution,
            unixtime=record.unixtime,
            fielddate=record.fielddate,
            edit_commit_id=record.edit_commit_id,
            edit_object_id=record.edit_object_id,
            um_login=record.um_login,
            moderation_status=record.moderation_status,
            role=record.role,
            object_path=DEFAULT_TREE_ROOT + '{}\t{}\t'.format(
                edits_constants.DEFINE_CATEGORY_GROUPS_DICT.get(convert_to_utf(record.object_category), 'None'),
                convert_to_utf(record.object_category)
            )
        )
