from maps.wikimap.stat.libs.common.lib.geobase_region import (
    geobase_id_by_shape,
    EARTH_REGION_ID,
    FILES,
    GEOBASE_JOB_MEMORY_LIMIT
)
from nile.api.v1 import (
    aggregators as na,
    extractors as ne,
    filters as nf
)
from qb2.api.v1 import (
    filters as qf
)
from yt.wrapper.ypath import ypath_join
from . import edit_processors

DEFAULT_REGION_TREE = '\t10000\t'
DEFAULT_NMAP_EDITS_PATH = '//home/logfeller/logs/nmaps-edits-log/1d'
DEFAULT_NMAP_USER_PATH = '//home/logfeller/logs/nmaps-users-dump-log/1d'
DEFAULT_NMAP_MODERATION_PATH = '//home/logfeller/logs/nmaps-moderation-log/1d'
DEFAULT_NMAP_ACL_ROLES_PATH = '//home/logfeller/logs/nmaps-acl-roles-dump/1d'


def make_fielddate(datefield):
    return str(edit_processors.convert_to_utf(datefield)).split(' ')[0]


def search_geoid_by_shape(geom):
    if geom:
        return str(geobase_id_by_shape(edit_processors.convert_to_utf(geom)))
    else:
        return EARTH_REGION_ID


def make_nmaps_edit_raws(job, date):
    '''
    Filtered nmaps-edits-log: action = object-modified

    :return:
    | puid | edit_commit_id | edit_object_id |
    |------+----------------+----------------|
    |  ... |            ... |            ... |
    '''
    return job.table(
        ypath_join(DEFAULT_NMAP_EDITS_PATH, date)
    ).project(
        'puid',
        'action',
        edit_raw_commit_id='commit_id',
        edit_raw_object_id='object_id'
    ).label('nmaps_edit_raw')


def make_nmaps_moderation_raws(job, date):
    '''
    Filtered nmaps-moderation-log: task_action != created

    :return:
    | puid | action | commit_id | object_id | object_category | resolution | unixtime | region_id | fielddate |
    |------+--------+-----------+-----------+-----------------+------------+----------+-----------+-----------|
    |  ... |    ... |       ... |       ... |             ... |        ... |      ... |       ... |       ... |
    '''
    return job.table(
        ypath_join(DEFAULT_NMAP_MODERATION_PATH, date)
    ).project(
        'puid',
        'action',
        'commit_id',
        'task_id',
        'object_id',
        'object_category',
        'resolution',
        unixtime=ne.custom(
            lambda unixtime: float(unixtime),
            'unixtime'
        ),
        region_id=ne.custom(search_geoid_by_shape, 'geom'),
        fielddate=ne.custom(make_fielddate, 'iso_eventtime'),
        files=FILES,
        memory_limit=GEOBASE_JOB_MEMORY_LIMIT
    ).label('nmaps_moderation_raw')


def make_nmaps_user_raw(job, date):
    '''
    :return:
    | puid | um_login | moderation_status |
    |------+----------+-------------------|
    |  ... |      ... |               ... |
    '''
    return job.table(
        ypath_join(DEFAULT_NMAP_USER_PATH, date)
    ).project(
        'puid',
        'um_login',
        'moderation_status'
    ).label('nmaps_user_raw')


def make_nmaps_acl_roles_raw(job, date):
    '''
    Filtered nmaps-acl-roles-dump: role expert or moderator
    :return:
    | puid | role |
    |------+------|
    |  ... |  ... |
    '''
    return job.table(
        ypath_join(DEFAULT_NMAP_ACL_ROLES_PATH, date)
    ).filter(
        nf.custom(
            lambda role: edit_processors.convert_to_utf(role).split('.')[0] in ('expert', 'moderator'),
            'role'
        )
    ).project(
        puid=ne.custom(
            lambda puid: str(puid)[:-1],
            'puid'
        ),
        acl_role='role'
    ).label('nmaps_acl_roles_raw')


def add_paths(job, nmaps_edits_raw, nmaps_moderation_raw, nmaps_user_raw, nmaps_acl_role_raw):
    '''
    Add trees for action, object_type and user_path ('yandex' for moderator, yandex-moderator, cartographer.
    'user' for others). Add field expert and moderator with roles

    :param nmaps_edits_raw:
    | puid | edit_commit_id | edit_object_id |
    |------+----------------+----------------|
    |  ... |            ... |            ... |

    :param nmaps_moderation_raw:
    | puid | action | commit_id | object_id | object_category | resolution | unixtime | region_id | fielddate |
    |------+--------+-----------+-----------+-----------------+------------+----------+-----------+-----------|
    |  ... |    ... |       ... |       ... |             ... |        ... |      ... |       ... |       ... |

    :param nmaps_user_raw:
    | puid | um_login | moderation_status |
    |------+----------+-------------------|
    |  ... |      ... |               ... |

    :param nmaps_acl_role_raw
    | puid | role |
    |------+------|
    |  ... |  ... |

    :param moderation_geo:
    | puid | fielddate | action | moderation_status | role | object_category | ... |
    |------+-----------+--------+-------------------+------+-----------------+-----|
    |  ... |       ... |    ... |               ... |  ... |             ... | ... |

    :return:
    | puid | fielddate | object_path | user_path | action_path | expert_role | moderator_role | ... |
    |------+-----------+-------------+-----------+-------------+-------------+----------------+-----|
    |  ... |       ... |         ... |       ... |         ... |         ... |            ... | ... |
    '''
    moderation_geo = nmaps_moderation_raw.join(
        nmaps_edits_raw,
        by=('puid', 'action'),
        type='left'
    ).join(
        nmaps_user_raw,
        by=('puid'),
        type='inner'
    ).join(
        nmaps_acl_role_raw,
        by=('puid'),
        type='left'
    ).project(
        ne.all(),
        role=ne.custom(
            lambda role: role or '',
            'acl_role'
        ),
        edit_commit_id=ne.custom(
            lambda edit_commit_id: edit_commit_id or None,
            'edit_raw_commit_id'
        ),
        edit_object_id=ne.custom(
            lambda edit_object_id: edit_object_id or None,
            'edit_raw_object_id'
        ),
    ).label('join_all_table')

    return moderation_geo.map(
        edit_processors.add_object_path_mapper
    ).label('create_object_path').map(
        edit_processors.add_user_path_mapper
    ).label('create_user_path').map(
        edit_processors.add_action_path_mapper
    ).label('create_action_path').project(
        ne.all(),
        expert_role=ne.custom(
            lambda role: 'Да' if edit_processors.convert_to_utf(role).split('.')[0] == 'expert' else None,
            'role'
        ),
        moderator_role=ne.custom(
            lambda role: 'Да' if edit_processors.convert_to_utf(role).split('.')[0] == 'moderator' else None,
            'role'
        )
    ).label('moderation_geo_with_paths')


def prepare_result_counts(stream):
    '''
    Calculating the result of statistics.
    Stream is aggregated by unique values from commit_id, object_id, edit_object_id, edit_commit_id, role and unixtime

    :param stream:
    | fielddate | puid | commit_id | ... | user_path | action_path | object_path | geo_path |
    |-----------+------+-----------+-----+-----------+-------------+-------------+----------|
    |       ... |  ... |       ... | ... |       ... |         ... |         ... |      ... |

    :return:
    | fielddate | geo_path | user_path | action_path | object_path | edited_objects | edits | ... | sum_total_cart_res |
    |-----------+----------+-----------+-------------+-------------+----------------+-------+-----+--------------------|
    |       ... |      ... |       ... |         ... |         ... |            ... |   ... | ... |                ... |
    '''
    count_edited_objects = stream.filter(
        qf.defined('edit_object_id')
    ).unique(
        'fielddate',
        'geo_path',
        'user_path',
        'action_path',
        'object_path',
        'edit_object_id'
    ).groupby(
        'fielddate',
        'geo_path',
        'user_path',
        'action_path',
        'object_path'
    ).aggregate(
        edited_objects=na.count()
    ).label('aggregate_by_edited_objects')

    count_edits = stream.filter(
        qf.defined('edit_commit_id')
    ).unique(
        'fielddate',
        'geo_path',
        'user_path',
        'action_path',
        'object_path',
        'edit_commit_id'
    ).groupby(
        'fielddate',
        'geo_path',
        'user_path',
        'action_path',
        'object_path'
    ).aggregate(
        edits=na.count()
    ).label('aggregate_by_edits')

    count_as_mod = stream.unique(
        'fielddate',
        'geo_path',
        'user_path',
        'action_path',
        'object_path',
        'commit_id'
    ).groupby(
        'fielddate',
        'geo_path',
        'user_path',
        'action_path',
        'object_path'
    ).aggregate(
        as_mod=na.count()
    ).label('aggregate_by_commit_id')

    count_as_mod_obj = stream.unique(
        'fielddate',
        'geo_path',
        'user_path',
        'action_path',
        'object_path',
        'object_id'
    ).groupby(
        'fielddate',
        'geo_path',
        'user_path',
        'action_path',
        'object_path'
    ).aggregate(
        as_mod_obj=na.count()
    ).label('aggregate_by_object_id')

    count_expert = stream.filter(
        nf.not_(nf.equals('expert_role', None))
    ).unique(
        'fielddate',
        'geo_path',
        'user_path',
        'action_path',
        'object_path',
        'expert_role'
    ).project(
        'fielddate',
        'geo_path',
        'user_path',
        'action_path',
        'object_path',
        expert='expert_role'
    ).label('aggregate_by_expert')

    count_moderator = stream.filter(
        nf.not_(nf.equals('moderator_role', None))
    ).unique(
        'fielddate',
        'geo_path',
        'user_path',
        'action_path',
        'object_path',
        'moderator_role'
    ).project(
        'fielddate',
        'geo_path',
        'user_path',
        'action_path',
        'object_path',
        moderator='moderator_role'
    ).label('aggregate_by_moderator')

    count_min_unixtime = stream.unique(
        'fielddate',
        'geo_path',
        'user_path',
        'action_path',
        'object_path',
        'unixtime'
    ).groupby(
        'fielddate',
        'geo_path',
        'user_path',
        'action_path',
        'object_path'
    ).aggregate(
        min_unixtime=na.min('unixtime')
    )

    count_max_unixtime = stream.unique(
        'fielddate',
        'geo_path',
        'user_path',
        'action_path',
        'object_path',
        'unixtime'
    ).groupby(
        'fielddate',
        'geo_path',
        'user_path',
        'action_path',
        'object_path'
    ).aggregate(
        max_unixtime=na.max('unixtime')
    )

    count_session = count_min_unixtime.join(
        count_max_unixtime,
        by=(
            'fielddate',
            'geo_path',
            'user_path',
            'action_path',
            'object_path'
        ),
        type='left'
    ).project(
        'fielddate',
        'geo_path',
        'user_path',
        'action_path',
        'object_path',
        session=ne.custom(
            lambda min_time, max_time: (max_time - min_time) / 3600,
            'min_unixtime',
            'max_unixtime'
        )
    ).label('aggregate_by_session')

    count_link = stream.unique(
        'fielddate',
        'geo_path',
        'user_path',
        'action_path',
        'object_path',
        'puid'
    ).groupby(
        'fielddate',
        'geo_path',
        'user_path',
        'action_path',
        'object_path'
    ).aggregate(
        link=na.max('puid')
    ).label('aggregate_by_link')

    count_uniq_tasks_id = stream.unique(
        'fielddate',
        'geo_path',
        'user_path',
        'action_path',
        'object_path',
        'task_id'
    ).groupby(
        'fielddate',
        'geo_path',
        'user_path',
        'action_path',
        'object_path'
    ).aggregate(
        count_uniq_task_id=na.count()
    ).label('aggregate_by_task_id')

    count_right_cart_res = stream.filter(
        nf.equals('moderation_status', b'cartographer'),
        nf.not_(nf.equals('resolution', b'revert'))
    ).unique(
        'fielddate',
        'geo_path',
        'user_path',
        'action_path',
        'object_path',
        'commit_id'
    ).groupby(
        'fielddate',
        'geo_path',
        'user_path',
        'action_path',
        'object_path'
    ).aggregate(
        sum_right_cart_res=na.count()
    ).label('aggregate_by_right_cart_res')

    count_total_cart_res = stream.filter(
        nf.equals('moderation_status', b'cartographer')
    ).unique(
        'fielddate',
        'geo_path',
        'user_path',
        'action_path',
        'object_path',
        'commit_id'
    ).groupby(
        'fielddate',
        'geo_path',
        'user_path',
        'action_path',
        'object_path'
    ).aggregate(
        sum_total_cart_res=na.count()
    ).label('aggregate_by_total_cart_res')

    return count_as_mod.join(
        count_as_mod_obj,
        by=(
            'fielddate',
            'geo_path',
            'user_path',
            'action_path',
            'object_path'
        ),
        type='left'
    ).join(
        count_edits,
        by=(
            'fielddate',
            'geo_path',
            'user_path',
            'action_path',
            'object_path'
        ),
        type='left'
    ).join(
        count_edited_objects,
        by=(
            'fielddate',
            'geo_path',
            'user_path',
            'action_path',
            'object_path'
        ),
        type='left'
    ).join(
        count_expert,
        by=(
            'fielddate',
            'geo_path',
            'user_path',
            'action_path',
            'object_path'
        ),
        type='left'
    ).join(
        count_moderator,
        by=(
            'fielddate',
            'geo_path',
            'user_path',
            'action_path',
            'object_path'
        ),
        type='left'
    ).join(
        count_session,
        by=(
            'fielddate',
            'geo_path',
            'user_path',
            'action_path',
            'object_path'
        ),
        type='left'
    ).join(
        count_link,
        by=(
            'fielddate',
            'geo_path',
            'user_path',
            'action_path',
            'object_path'
        ),
        type='left'
    ).join(
        count_uniq_tasks_id,
        by=(
            'fielddate',
            'geo_path',
            'user_path',
            'action_path',
            'object_path'
        ),
        type='left'
    ).join(
        count_right_cart_res,
        by=(
            'fielddate',
            'geo_path',
            'user_path',
            'action_path',
            'object_path'
        ),
        type='left'
    ).join(
        count_total_cart_res,
        by=(
            'fielddate',
            'geo_path',
            'user_path',
            'action_path',
            'object_path'
        ),
        type='left'
    ).label(
        'join_all_calculations'
    ).project(
        'fielddate',
        'geo_path',
        'user_path',
        'action_path',
        'object_path',
        auto_simple=ne.const(0),
        auto_hard=ne.const(0),
        edited_non_auto=ne.const(0),
        edited_objects=ne.custom(
            lambda edited_objects: edited_objects or 0,
            'edited_objects'
        ),
        edits=ne.custom(
            lambda edits: edits or 0,
            'edits'
        ),
        as_mod=ne.custom(
            lambda as_mod: as_mod or 0,
            'as_mod'
        ),
        as_mod_obj=ne.custom(
            lambda as_mod_obj: as_mod_obj or 0,
            'as_mod_obj'
        ),
        expert=ne.custom(
            lambda expert: expert or None,
            'expert'
        ),
        moderator=ne.custom(
            lambda moderator: moderator or None,
            'moderator'
        ),
        session=ne.custom(
            lambda session: session or 0,
            'session'
        ),
        link=ne.custom(
            lambda link: link or 0,
            'link'
        ),
        count_uniq_task_id=ne.custom(
            lambda count_uniq_task_id: count_uniq_task_id or 0,
            'count_uniq_task_id'
        ),
        sum_right_cart_res=ne.custom(
            lambda sum_right_cart_res: sum_right_cart_res or 0,
            'sum_right_cart_res'
        ),
        sum_total_cart_res=ne.custom(
            lambda sum_total_cart_res: sum_total_cart_res or 0,
            'sum_total_cart_res'
        )
    ).label('prepare_result_counts')


def set_region_tree(stream, major_regions):
    return stream.join(
        major_regions,
        by='region_id',
        type='left',
        assume_small_right=True,
        allow_undefined_keys=False,
        assume_defined=True,
        memory_limit=8 * 1024
    ).project(
        ne.all(),
        geo_path=ne.custom(
            lambda rt: rt or DEFAULT_REGION_TREE,
            'region_tree'
        )
    ).label('add_geo_path')
