from nile.api.v1 import Record

from . import edits_constants


DEFAULT_TREE_ROOT = '\t_total_\t'


def convert_to_utf(field):
    if type(field) == bytes:
        return field.decode('utf-8')
    else:
        return field


def add_event_path_mapper(records):
    '''
    Create tree for event_path.

    records
    | puid | task_id | region_id | fielddate | object_path | event_type |
    |------+---------+-----------+-----------+-------------+------------|
    |  ... |     ... |       ... |       ... |         ... |        ... |

    result
    | puid | task_id | region_id | fielddate | object_path |          event_path |
    |------+---------+-----------+-----------+-------------+---------------------|
    |  ... |     ... |       ... |       ... |         ... |       '\t_total_\t' |
    |  ... |     ... |       ... |       ... |         ... | '\t_total_\tedit\t' |
    |  ... |     ... |       ... |       ... |         ... |                 ... |
    '''
    for record in records:
        yield Record(
            puid=record.puid,
            task_id=record.task_id,
            region_id=record.region_id,
            fielddate=record.fielddate,
            object_path=record.object_path,
            event_path=DEFAULT_TREE_ROOT
        )
        yield Record(
            puid=record.puid,
            task_id=record.task_id,
            region_id=record.region_id,
            fielddate=record.fielddate,
            object_path=record.object_path,
            event_path=DEFAULT_TREE_ROOT + '{}\t'.format(convert_to_utf(record.event_type))
        )


def add_object_path_mapper(records):
    '''
    Create tree for object_path.
    Take group from dict in edits_constats.DEFINE_CATEGORY_GROUPS_DICT

    records
    | puid | task_id | region_id | fielddate | object_category | event_type |
    |------+---------+-----------+-----------+-----------------+------------|
    |  ... |     ... |       ... |       ... |             ... |        ... |

    result
    | puid | task_id | region_id | fielddate | event_type |                    object_path |
    |------+---------+-----------+-----------+------------+--------------------------------|
    |  ... |     ... |       ... |       ... |        ... |                  '\t_total_\t' |
    |  ... |     ... |       ... |       ... |        ... |        '\t_total_\trd_group\t' |
    |  ... |     ... |       ... |       ... |        ... | '\t_total_\trd_group\trd_jc\t' |
    |  ... |     ... |       ... |       ... |        ... |                            ... |
    '''
    for record in records:
        yield Record(
            puid=record.puid,
            task_id=record.task_id,
            event_type=record.event_type,
            region_id=record.region_id,
            fielddate=record.fielddate,
            object_path=DEFAULT_TREE_ROOT
        )
        yield Record(
            puid=record.puid,
            task_id=record.task_id,
            event_type=record.event_type,
            region_id=record.region_id,
            fielddate=record.fielddate,
            object_path=DEFAULT_TREE_ROOT + '{}\t'.format(
                edits_constants.DEFINE_CATEGORY_GROUPS_DICT.get(convert_to_utf(record.object_category), 'None')
            )
        )
        yield Record(
            puid=record.puid,
            task_id=record.task_id,
            event_type=record.event_type,
            region_id=record.region_id,
            fielddate=record.fielddate,
            object_path=DEFAULT_TREE_ROOT + '{}\t{}\t'.format(
                edits_constants.DEFINE_CATEGORY_GROUPS_DICT.get(convert_to_utf(record.object_category), 'None'),
                convert_to_utf(record.object_category)
            )
        )
