from nile.api.v1 import (
    aggregators as na,
    extractors as ne,
    filters as nf
)
from yt.wrapper.ypath import ypath_join

from . import edit_processors

DEFAULT_REGION_TREE = '\t10000\t'
DEFAULT_NMAP_EDITS_PATH = '//home/logfeller/logs/nmaps-edits-log/1d'
DEFAULT_NMAP_USER_PATH = '//home/logfeller/logs/nmaps-users-dump-log/1d'
DEFAULT_NMAP_MODERATION_PATH = '//home/logfeller/logs/nmaps-moderation-log/1d'


def make_fielddate(datefield):
    return str(edit_processors.convert_to_utf(datefield)).split(' ')[0]


def make_revert_reason(job, date):
    '''
    Filtered nmaps-moderation-log

    result:
    | puid | commit_id | object_category | event_type | fielddate |
    |------+-----------+-----------------+------------+-----------|
    |  ... |       ... |             ... |        ... |       ... |
    '''
    return job.table(
        ypath_join(DEFAULT_NMAP_MODERATION_PATH, date)
    ).filter(
        nf.equals('resolution', b'revert')
    ).project(
        'puid',
        'commit_id',
        'object_category',
        'revert_reason',
        fielddate=ne.custom(make_fielddate, 'iso_eventtime')
    ).label('revert_reason')


def make_nmaps_user_raw(job, date):
    '''
    result
    | puid | moderation_status |
    |------+-------------------|
    |  ... |               ... |
    '''
    return job.table(
        ypath_join(DEFAULT_NMAP_USER_PATH, date)
    ).project(
        'puid',
        'moderation_status'
    ).label('nmaps_user_raw')


def add_paths(job, revert_reason_raw, nmaps_user_raw):
    '''
    Add trees for event_type, object_type and add user status (moderator, yandex-moderator, cartographer)

    revert_reason_raw:
    | puid | commit_id | object_category | event_type | region_id | fielddate |
    |------+---------+-----------------+------------+-----------+-----------|
    |  ... |     ... |             ... |        ... |       ... |       ... |

    nmaps_user_raw:
    | puid | moderation_status |
    |------+-------------------|
    |  ... |               ... |

    revert_reason:
    | commit_id | fielddate | puid | revert_path | object_path |
    |-----------+-----------+------+-------------+-------------|
    |       ... |       ... |  ... |         ... |         ... |

    result:
    | commit_id | fielddate | puid | revert_path | object_path | status_path |
    |-----------+-----------+------+-------------+-------------+-------------|
    |       ... |       ... |  ... |         ... |         ... |         ... |
    '''
    revert_reason = revert_reason_raw.map(
        edit_processors.add_object_path_mapper
    ).label('create_object_path').map(
        edit_processors.add_revert_path_mapper
    ).label('create_revert_path')

    return revert_reason.join(
        nmaps_user_raw,
        by=('puid'),
        type='inner'
    ).map(
        edit_processors.add_status_path_mapper
    ).label('revert_reason_with_paths')


def prepare_result_counts(stream):
    '''
    Calculating the result of statistics.
    Stream is aggregated by 3 unique values: commit_id, object_id, and puid.

    stream
    | puid | commit_id | revert_path | fielddate | object_path | status_path |
    |------+-----------+-------------+-----------+-------------+-------------|
    |  ... |       ... |         ... |       ... |         ... |         ... |

    result
    | fielddate | status_path | revert_path | object_path | count_uniq_commit_id |
    |-----------+-------------+-------------+-------------+----------------------|
    |       ... |         ... |         ... |         ... |                  ... |
    '''
    return stream.unique(
        'fielddate',
        'revert_path',
        'object_path',
        'status_path',
        'commit_id'
    ).label(
        'uniq_results'
    ).groupby(
        'fielddate',
        'revert_path',
        'object_path',
        'status_path'
    ).aggregate(
        count_uniq_commit_id=na.count()
    ).label(
        'count_results'
    ).project(
        'fielddate',
        'revert_path',
        'object_path',
        'status_path',
        count_uniq_commit_id=ne.custom(
            lambda count_uniq_commit_id: count_uniq_commit_id or 0,
            'count_uniq_commit_id'
        )
    ).label('prepare_result_counts')
