from nile.api.v1 import Record

from . import edits_constants


DEFAULT_TREE_ROOT = '\t_total_\t'


def convert_to_utf(field):
    if type(field) == bytes:
        return field.decode('utf-8')
    else:
        return field


def make_event_path(vars, path):
    '''
    Create tree for events. Data from the path and vars fields are combined
    Example: '\tnmaps_www\tcreate_geoobject\theader\t'
    First and second element from path field. Other elements - data from vars field
    '''
    path = convert_to_utf(path)
    tree_string = '\t'
    for element_path in path.split('.'):
        tree_string += '{}\t'.format(element_path)
        yield tree_string

    if vars:
        dict_vars = dict(convert_to_utf(vars))

        if 'path' in dict_vars:
            for element_event_path in dict_vars['path'].lstrip('/').split('/'):
                tree_string += '{}\t'.format(element_event_path)
                yield tree_string
        elif 'button' in dict_vars:
            tree_string += '{}\t'.format('button')
            yield tree_string
            tree_string += '{}\t'.format(dict_vars['button'])
            yield tree_string
        elif 'newZoom' in dict_vars:
            tree_string += '{}\t'.format('bound_change')
            yield tree_string
            if dict_vars['newZoom'] == dict_vars['oldZoom']:
                tree_string += '{}\t'.format('drug')
                yield tree_string
            else:
                tree_string += '{}\t'.format('zoom')
                yield tree_string
        elif 'item' in dict_vars:
            tree_string += '{}\t'.format('item')
            yield tree_string
            if 'checked' in dict_vars and dict_vars['checked']:
                tree_string += '{}\t'.format(dict_vars['checked'])
                yield tree_string
        else:
            pass


def make_full_user_path(edit_field, status_field):
    '''
    Create full tree of status_path
    New users (with less than fifteen edits) are placed in the \\ttotal\\tusers\\t group,
    but users with status_field from edits_constants.YANDEX_USERS are placed in the \\ttotal\\tyandex\\t group
    '''
    tree_string = DEFAULT_TREE_ROOT
    yield tree_string

    edit_field = convert_to_utf(edit_field)
    status_field = convert_to_utf(status_field)
    if convert_to_utf(status_field) in edits_constants.YANDEX_USERS:
        tree_string += 'yandex\t'
        yield tree_string
    else:
        tree_string += 'users\t'
        yield tree_string
        if status_field == 'common':
            tree_string += 'common\t'
            yield tree_string

    if edit_field == edits_constants.NEWBIE1_EDITS_COUNT:
        tree_string += 'new_1commit\t'
        yield tree_string
    elif edits_constants.NEWBIE1_EDITS_COUNT < edit_field <= edits_constants.NEWBIE_EDITS_COUNT_MAX:
        tree_string += 'new\t'
        yield tree_string
    elif not(edit_field) and status_field == 'common':
        tree_string += 'no_edits\t'
        yield tree_string
    elif not(status_field):
        tree_string += 'None\t'
        yield tree_string
    else:
        tree_string += '{}\t'.format(status_field)
        yield tree_string


def add_user_path_mapper(records):
    '''
    Create tree for status_path.
    Mark users for yandex-moderators and users
    + mark users with a small number of edits (up to 15)

    records
    | edits | puid | moderation_status |
    |-------+------+-------------------|
    |   ... |  ... |               ... |

    result
    | puid |                         status_path |
    |------+-------------------------------------|
    |  ... |                       '\t_total_\t' |
    |  ... |               '\t_total_\tyandex\t' |
    |  ... | '\t_total_\tyandex\tcartographer\t' |
    |  ... |                                 ... |
    '''
    for record in records:
        for element_user_path in make_full_user_path(record.edits, record.moderation_status):
            yield Record(
                puid=record.puid,
                user_path=element_user_path
            )


def add_event_path_mapper(records):
    '''
    Create tree for event_path.
    Use data from 2 fields: vars (type: dict) and path (type: string)

    records
    | session_id | puid | ... | vars | path |
    |------------+------+-----+------+------|
    |        ... |  ... | ... |  ... |  ... |

    result
    | puid | ... |                                event_path |
    |------+-----+-------------------------------------------|
    |  ... |     |                           '\tnmaps_www\t' |
    |  ... |     |         '\tnmaps_www\tcreate_geoobject\t' |
    |  ... |     | '\tnmaps_www\tcreate_geoobject\theader\t' |
    |  ... |     |                                       ... |
    '''
    for record in records:
        for element_event_path in make_event_path(record.vars, record.path):
            yield Record(
                fielddate=record.fielddate,
                puid=record.puid,
                session_id=record.session_id,
                click=record.click,
                shows=record.shows,
                region_id=record.region_id,
                puid_click=record.puid_click,
                session_click=record.session_click,
                event_path=element_event_path
            )
