from nile.api.v1 import (
    aggregators as na,
    extractors as ne
)
from qb2.api.v1 import (
    filters as qf
)
from yt.wrapper.ypath import ypath_join

from . import edit_processors
from . import edits_constants

DEFAULT_REGION_TREE = '\t10000\t'
DEFAULT_NMAP_EDITS_PATH = '//home/logfeller/logs/nmaps-edits-log/1d'
DEFAULT_NMAP_USER_PATH = '//home/logfeller/logs/nmaps-users-dump-log/1d'
DEFAULT_BEBR_LOG = '//home/logfeller/logs/redir-log/1d'


def make_fielddate(datefield):
    return edit_processors.convert_to_utf(datefield).split(' ')[0]


def make_bebr_table_with_filter(job, date):
    '''
    Create normalized and filtered table from DEFAULT_BEBR_LOG
    DEFAULT_BEBR_LOG
    | key | subkey | value |
    |-------+------+-------|
    |   ... |  ... |   ... |

    result:
    | dtype | event_type | path | pid | puid | session_id | fielddate | vars | geo_id | region_id | puid_click |   session_click | click | shows | event_path |
    |-------+------------+------+-----+------+------------+-----------+------+--------+-----------+------------+-----------------+-------+-------+------------|
    |   ... |        ... |  ... | ... |  ... |        ... |       ... |  ... |    ... |    geo_id |  puid/None | session_id/None |   ... |   ... |        ... |
    '''
    redir_stream = job.table(
        ypath_join(DEFAULT_BEBR_LOG, date)
    ).qb2(
        log='redir-log-bebr',
        fields=(
            'date',
            'event_type',
            'path',
            'pid',
            'puid',
            'session_id',
            'timestamp',
            'vars',
            'geo_id'
        ),
        filters=(
            qf.equals('dtype', b'bebr'),
            qf.equals('pid', b'458'),
            qf.defined('ip'),
            qf.defined('event')
        ),
        mode='yamr_lines',
        intensity='large_data',
    ).label(
        'filter_bebr_log'
    ).project(
        ne.all(),
        fielddate='date',
        region_id=ne.custom(lambda geo_id: str(geo_id), 'geo_id'),
        puid_click=ne.custom(
            lambda puid, event_type: puid if edit_processors.convert_to_utf(event_type) in ('click', 'close') and puid else None,
            'puid',
            'event_type'
        ),
        session_click=ne.custom(
            lambda session_id, event_type: session_id if edit_processors.convert_to_utf(event_type) == 'click' and session_id else None,
            'session_id',
            'event_type'
        ),
        click=ne.custom(
            lambda event_type, puid, path: 1 if edit_processors.convert_to_utf(event_type) == 'click' and puid and path else None,
            'event_type',
            'puid',
            'path'
        ),
        shows=ne.custom(
            lambda event_type, puid, path: 1 if edit_processors.convert_to_utf(event_type) == 'show' and puid and path else None,
            'event_type',
            'puid',
            'path'
        )
    ).label(
        'custom_bebr_log'
    )
    return redir_stream


def make_edits_count_per_user(job, date):
    '''
    result:
    | edits | puid |
    |-------+------|
    |   ... |  ... |
    '''
    dates = '{' + edits_constants.DATE_START_EDIT_LOG + '..' + str(date) + '}'
    return job.table(
        ypath_join(DEFAULT_NMAP_EDITS_PATH, dates),
        ignore_missing=True
    ).groupby('puid').aggregate(
        edits=na.count()
    ).label('edits_count_per_user')


def make_nmaps_edits(job, date):
    '''
    result
    | puid | action_sum_create |
    |------+-------------------|
    |  ... |               ... |
    '''
    return job.table(
        ypath_join(DEFAULT_NMAP_EDITS_PATH, date)
    ).label(
        'nmaps_edit_raw'
    ).project(
        'puid',
        action_create=ne.custom(lambda action: 1 if edit_processors.convert_to_utf(action) == 'object-created' else 0, 'action')
    ).label(
        'add_action_create'
    ).groupby(
        'puid'
    ).aggregate(
        action_sum_create=na.sum('action_create')
    ).label('nmaps_edits')


def make_nmaps_user_raw(job, date):
    '''
    result
    | puid | moderation_status |
    |------+-------------------|
    |  ... |               ... |
    '''
    return job.table(
        ypath_join(DEFAULT_NMAP_USER_PATH, date)
    ).project(
        'puid',
        'moderation_status'
    ).label('nmaps_user_raw')


def add_user_path(job, edits_count_per_user, nmaps_user_raw):
    '''
    Add trees for action with objects, user status and object groups

    edits_count_per_user:
    | edits | puid |
    |-------+------|
    |    .. |  ... |

    nmaps_user_raw:
    | puid | moderation_status |
    |------+-------------------|
    |  ... |               ... |

    nmaps_users_status:
    | puid |   user_path |
    |------+-------------|
    |  ... |         ... |
    '''
    return edits_count_per_user.join(
        nmaps_user_raw,
        by='puid',
        type='left'
    ).label(
        'join_for_user_path'
    ).map(
        edit_processors.add_user_path_mapper
    ).label('create_user_path')


def add_event_path(job, bebr_filter_table_raw):
    '''
    Add trees for action with objects, user status and object groups

    bebr_filter_table
    | puid |  ... | event_path |
    |------+------+------------|
    |  ... |  ... |        ... |
    '''

    return bebr_filter_table_raw.map(
        edit_processors.add_event_path_mapper
    ).label(
        'create_event_path'
    )


def make_nmaps_edits_for_count(job, nmaps_edits, bebr_filter_table, nmaps_users_status):
    '''
    Create aggregate big table with all information about edits.

    nmaps_edits:
    | puid | fielddate | action_sum_create |
    |------+-----------+-------------------|
    |  ... |       ... |               ... |

    bebr_filter_table
    | dtype | event_type | path | pid | puid | session_id | fielddate | vars | geo_id | region_id | puid_click |   session_click | click | shows | event_path |
    |-------+------------+------+-----+------+------------+-----------+------+--------+-----------+------------+-----------------+-------+-------+------------|
    |   ... |        ... |  ... | ... |  ... |        ... |       ... |  ... |    ... |    geo_id |  puid/None | session_id/None |   ... |   ... |        ... |

    nmaps_users_status:
    | puid |   user_path |
    |------+-------------|
    |  ... |         ... |

    Result:
    | puid | ... | action_sum_create | user_path | puid_created |
    |------+-----+-------------------+-----------+--------------|
    |  ... | ... |               ... |       ... |          ... |
    '''

    return bebr_filter_table.join(
        nmaps_edits,
        by='puid',
        type='left'
    ).label(
        'add_nmaps_edits_to_bebr'
    ).join(
        nmaps_users_status,
        by='puid',
        type='inner'
    ).project(
        ne.all(),
        puid_created=ne.custom(
            lambda action_sum_create, puid: puid if action_sum_create else None,
            'action_sum_create',
            'puid'
        )
    ).label(
        'make_nmaps_edits_for_count'
    )


def prepare_result_counts(stream):
    '''
    Calculating the result of statistics.
    Stream is aggregated by 3 unique values: commit_id, object_id, and puid.

    stream
    | fielddate | puid | ... | event_path | user_path | puid_created | region_tree |
    |-----------+------+-----+------------+-----------+--------------+-------------|
    |       ... |  ... | ... |        ... |       ... |          ... |         ... |

    stream_count_uniq_puid
    | fielddate | event_path | region_tree | user_path | count_uniq_puid |
    |-----------+------------+-------------+-----------+-----------------|
    |       ... |        ... |         ... |       ... |             ... |

    stream_count_uniq_puid_click
    | fielddate | event_path | region_tree | user_path | count_uniq_puid_click |
    |-----------+------------+-------------+-----------+-----------------------|
    |       ... |        ... |         ... |       ... |                   ... |

    stream_count_uniq_puid_edits
    | fielddate | event_path | region_tree | user_path | count_uniq_puid_edits |
    |-----------+------------+-------------+-----------+-----------------------|
    |       ... |        ... |         ... |       ... |                   ... |

    stream_count_uniq_puid_created
    | fielddate | event_path | region_tree | user_path | count_uniq_puid_created |
    |-----------+------------+-------------+-----------+-------------------------|
    |       ... |        ... |         ... |       ... |                     ... |

    stream_count_uniq_session_id
    | fielddate | event_path | region_tree | user_path | count_uniq_session_id |
    |-----------+------------+-------------+-----------+-----------------------|
    |       ... |        ... |         ... |       ... |                   ... |

    stream_count_uniq_session_click
    | fielddate | event_path | region_tree | user_path | count_uniq_session_click |
    |-----------+------------+-------------+-----------+--------------------------|
    |       ... |        ... |         ... |       ... |                      ... |

    stream_count_sum_click
    | fielddate | event_path | region_tree | user_path | sum_click |
    |-----------+------------+-------------+-----------+-----------|
    |       ... |        ... |         ... |       ... |       ... |

    stream_count_sum_show
    | fielddate | event_path | region_tree | user_path | sum_show |
    |-----------+------------+-------------+-----------+----------|
    |       ... |        ... |         ... |       ... |      ... |

    result
    | fielddate | ... | count_uniq_puid | count_uniq_puid_click | count_uniq_puid_edits | count_uniq_puid_created | count_uniq_session_id | count_uniq_session_click | sum_click | sum_show |
    |-----------+-----+-----------------+-----------------------+-----------------------+-------------------------+-----------------------+--------------------------+-----------+----------|
    |       ... | ... |             ... |                   ... |                   ... |                     ... |                   ... |                      ... |       ... |      ... |
    '''
    stream_count_uniq_puid = stream.unique(
        'fielddate',
        'event_path',
        'region_tree',
        'user_path',
        'puid'
    ).groupby(
        'fielddate',
        'event_path',
        'region_tree',
        'user_path'
    ).aggregate(
        count_uniq_puid=na.count()
    ).label('aggregate_by_puid')

    stream_count_uniq_puid_click = stream.filter(
        qf.defined('puid_click')
    ).unique(
        'fielddate',
        'event_path',
        'region_tree',
        'user_path',
        'puid_click'
    ).groupby(
        'fielddate',
        'event_path',
        'region_tree',
        'user_path'
    ).aggregate(
        count_uniq_puid_click=na.count()
    ).label('aggregate_by_puid_click')

    stream_count_uniq_puid_edits = stream.filter(
        qf.defined('action_sum_create')
    ).unique(
        'fielddate',
        'event_path',
        'region_tree',
        'user_path',
        'puid'
    ).groupby(
        'fielddate',
        'event_path',
        'region_tree',
        'user_path'
    ).aggregate(
        count_uniq_puid_edits=na.count()
    ).label('aggregate_by_puid_edits')

    stream_count_uniq_puid_created = stream.unique(
        'fielddate',
        'event_path',
        'region_tree',
        'user_path',
        'puid_created'
    ).groupby(
        'fielddate',
        'event_path',
        'region_tree',
        'user_path'
    ).aggregate(
        count_uniq_puid_created=na.count()
    ).label('aggregate_by_puid_created')

    stream_count_uniq_session_id = stream.unique(
        'fielddate',
        'event_path',
        'region_tree',
        'user_path',
        'session_id'
    ).groupby(
        'fielddate',
        'event_path',
        'region_tree',
        'user_path'
    ).aggregate(
        count_uniq_session_id=na.count()
    ).label('aggregate_by_session_id')

    stream_count_uniq_session_click = stream.filter(
        qf.defined('session_click')
    ).unique(
        'fielddate',
        'event_path',
        'region_tree',
        'user_path',
        'session_click'
    ).groupby(
        'fielddate',
        'event_path',
        'region_tree',
        'user_path'
    ).aggregate(
        count_uniq_session_click=na.count()
    ).label('aggregate_by_session_click')

    stream_count_sum_click = stream.filter(
        qf.defined('click')
    ).groupby(
        'fielddate',
        'event_path',
        'region_tree',
        'user_path'
    ).aggregate(
        sum_click=na.count()
    ).label('aggregate_bu_sum_click')

    stream_count_sum_show = stream.filter(
        qf.defined('shows')
    ).groupby(
        'fielddate',
        'event_path',
        'region_tree',
        'user_path'
    ).aggregate(
        sum_show=na.count()
    ).label('aggregate_bu_sum_show')

    return stream_count_uniq_puid.join(
        stream_count_uniq_puid_click,
        by=('fielddate',
            'event_path',
            'region_tree',
            'user_path'
            ),
        type='left'
    ).join(
        stream_count_uniq_puid_edits,
        by=('fielddate',
            'event_path',
            'region_tree',
            'user_path'
            ),
        type='left'
    ).join(
        stream_count_uniq_puid_created,
        by=('fielddate',
            'event_path',
            'region_tree',
            'user_path'
            ),
        type='left'
    ).join(
        stream_count_uniq_session_id,
        by=('fielddate',
            'event_path',
            'region_tree',
            'user_path'
            ),
        type='left'
    ).join(
        stream_count_uniq_session_click,
        by=('fielddate',
            'event_path',
            'region_tree',
            'user_path'
            ),
        type='left'
    ).join(
        stream_count_sum_click,
        by=('fielddate',
            'event_path',
            'region_tree',
            'user_path'
            ),
        type='left'
    ).join(
        stream_count_sum_show,
        by=('fielddate',
            'event_path',
            'region_tree',
            'user_path'
            ),
        type='left'
    ).project(
        'fielddate',
        'event_path',
        'user_path',
        count_uniq_puid=ne.custom(
            lambda count_uniq_puid: count_uniq_puid or 0,
            'count_uniq_puid'
        ),
        count_uniq_puid_click=ne.custom(
            lambda count_uniq_puid_click: count_uniq_puid_click or 0,
            'count_uniq_puid_click'
        ),
        count_uniq_puid_edits=ne.custom(
            lambda count_uniq_puid_edits: count_uniq_puid_edits or 0,
            'count_uniq_puid_edits'
        ),
        count_uniq_puid_created=ne.custom(
            lambda count_uniq_puid_created: count_uniq_puid_created or 0,
            'count_uniq_puid_created'
        ),
        count_uniq_session_id=ne.custom(
            lambda count_uniq_session_id: count_uniq_session_id or 0,
            'count_uniq_session_id'
        ),
        count_uniq_session_click=ne.custom(
            lambda count_uniq_session_click: count_uniq_session_click or 0,
            'count_uniq_session_click'
        ),
        sum_click=ne.custom(
            lambda sum_click: sum_click or 0,
            'sum_click'
        ),
        sum_show=ne.custom(
            lambda sum_show: sum_show or 0,
            'sum_show'
        ),
        geo_path='region_tree'
    ).label('prepare_result_counts')


def set_region_tree(stream, major_regions):
    return stream.join(
        major_regions,
        by='region_id',
        type='left',
        assume_small_right=True,
        allow_undefined_keys=False,
        assume_defined=True,
        memory_limit=8 * 1024
    ).project(
        ne.all(),
        region_tree=ne.custom(
            lambda rt: rt or DEFAULT_REGION_TREE,
            'region_tree'
        )
    )
