from nile.api.v1 import Record

from . import edits_constants


DEFAULT_TREE_ROOT = '\t_total_\t'


def convert_to_utf(field):
    return field.decode('utf-8')


def make_group_status_path(status_field):
    '''
    Create group name for tree of status_path.
    '''
    status_field = convert_to_utf(status_field)
    return DEFAULT_TREE_ROOT + \
        ('yandex\t' if status_field in edits_constants.YANDEX_USERS else 'users\t')


def make_full_status_path(edit_field, status_field):
    '''
    Create full tree of status_path
    New users (with less than fifteen edits) are placed in the \\ttotal\\tusers\\t group,
    but users with status_field from edits_constants.YANDEX_USERS are placed in the \\ttotal\\tyandex\\t group
    '''
    tree_string = make_group_status_path(status_field)

    status_field = convert_to_utf(status_field)

    if edit_field == edits_constants.NEWBIE1_EDITS_COUNT:
        tree_string += 'new_1commit\t'
    elif edits_constants.NEWBIE1_EDITS_COUNT < edit_field <= edits_constants.NEWBIE_EDITS_COUNT_MAX:
        tree_string += 'new\t'
    else:
        tree_string += '{}\t'.format(status_field)

    return tree_string


def add_status_path_mapper(records):
    '''
    Create tree for status_path.
    Mark users for yandex-moderators and users
    + mark users with a small number of edits (up to 15)

    records
    | edits | puid | moderation_status |
    |-------+------+-------------------|
    |   ... |  ... |               ... |

    result
    | puid |                         status_path |
    |------+-------------------------------------|
    |  ... |                       '\t_total_\t' |
    |  ... |               '\t_total_\tyandex\t' |
    |  ... | '\t_total_\tyandex\tcartographer\t' |
    |  ... |                                 ... |
    '''
    for record in records:
        yield Record(
            puid=record.puid,
            status_path=DEFAULT_TREE_ROOT
        )
        yield Record(
            puid=record.puid,
            status_path=make_group_status_path(record.moderation_status)
        )
        yield Record(
            puid=record.puid,
            status_path=make_full_status_path(record.edits, record.moderation_status)
        )


def add_action_path_mapper(records):
    '''
    Create tree for action_path.

    records
    | action | commit_id | geom | object_category | object_id | puid | fielddate | region_id |
    |--------+-----------+------+-----------------+-----------+------+-----------+-----------|
    |    ... |       ... |  ... |             ... |       ... |  ... |       ... |       ... |

    result
    | commit_id | object_id | puid | fielddate | region_id |                            action_path |
    |-----------+-----------+------+-----------+-----------+----------------------------------------|
    |       ... |       ... |  ... |       ... |       ... |                          '\t_total_\t' |
    |       ... |       ... |  ... |       ... |       ... |         '\t_total_\tobject-modified\t' |
    |       ... |       ... |  ... |       ... |       ... |                                    ... |
    '''
    for record in records:
        yield Record(
            commit_id=record.commit_id,
            region_id=record.region_id,
            fielddate=record.fielddate,
            puid=record.puid,
            object_id=record.object_id,
            action_path=DEFAULT_TREE_ROOT
        )
        yield Record(
            commit_id=record.commit_id,
            region_id=record.region_id,
            fielddate=record.fielddate,
            puid=record.puid,
            object_id=record.object_id,
            action_path=DEFAULT_TREE_ROOT + '{}\t'.format(convert_to_utf(record.action))
        )


def add_object_path_mapper(records):
    '''
    Create tree for object_path.
    Take group from dict in edits_constats.DEFINE_CATEGORY_GROUPS_DICT

    records
    | action | commit_id | geom | object_category | object_id | puid | fielddate | region_id |
    |--------+-----------+------+-----------------+-----------+------+-----------+-----------|
    |    ... |       ... |  ... |             ... |       ... |  ... |       ... |       ... |

    result
    | commit_id | object_id | puid | fielddate | region_id |                            object_path |
    |-----------+-----------+------+-----------+-----------+----------------------------------------|
    |       ... |       ... |  ... |       ... |       ... |                          '\t_total_\t' |
    |       ... |       ... |  ... |       ... |       ... |                '\t_total_\trd_group\t' |
    |       ... |       ... |  ... |       ... |       ... |         '\t_total_\trd_group\trd_jc\t' |
    |       ... |       ... |  ... |       ... |       ... |                                    ... |
    '''
    for record in records:
        yield Record(
            commit_id=record.commit_id,
            region_id=record.region_id,
            fielddate=record.fielddate,
            puid=record.puid,
            object_id=record.object_id,
            object_path=DEFAULT_TREE_ROOT
        )
        yield Record(
            commit_id=record.commit_id,
            region_id=record.region_id,
            fielddate=record.fielddate,
            puid=record.puid,
            object_id=record.object_id,
            object_path=DEFAULT_TREE_ROOT + '{}\t'.format(
                edits_constants.DEFINE_CATEGORY_GROUPS_DICT.get(convert_to_utf(record.object_category), 'None')
            )
        )
        yield Record(
            commit_id=record.commit_id,
            region_id=record.region_id,
            fielddate=record.fielddate,
            puid=record.puid,
            object_id=record.object_id,
            object_path=DEFAULT_TREE_ROOT + '{}\t{}\t'.format(
                edits_constants.DEFINE_CATEGORY_GROUPS_DICT.get(convert_to_utf(record.object_category), 'None'),
                convert_to_utf(record.object_category)
            )
        )
