import common
from nile.api.v1 import (
    local,
    Record
)
import maps.wikimap.stat.nmaps_edits.lib.edits_constants as edits_constants


def test_create_category_groups_dict():
    define_category_groups_dict = {
        'rd': 'rd_group',
        'rd_el': 'rd_group',
        'rd_jc': 'rd_group',
        'cond': 'cond_group',
        'cond_annotation': 'cond_group',
        'cond_cam': 'cond_group',
        'ad': 'ad_group',
        'ad_subst': 'ad_group',
        'ad_cnt': 'ad_group',
        'ad_jc': 'ad_group',
        'bld': 'bld_group',
        'bld_complex': 'bld_group',
    }

    test_xml_string = '''<?xml version="1.0" encoding="UTF-8"?>
<category-groups xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xsi:noNamespaceSchemaLocation="categories.xsd">
    <category-group id="rd_group" ugc_valuable="true">
        <label>{{category-groups:rd_group}}</label>
        <categories>
            <category id="rd"/>
            <category id="rd_el"/>
            <category id="rd_jc"/>
        </categories>
    </category-group>
    <category-group id="cond_group" ugc_valuable="true">
        <label>{{category-groups:cond_group}}</label>
        <categories>
            <category id="cond"/>
            <category id="cond_annotation"/>
            <category id="cond_cam"/>
        </categories>
    </category-group>
    <category-group id="ad_group" ugc_valuable="true">
        <label>{{category-groups:ad_group}}</label>
        <categories>
            <category id="ad"/>
            <category id="ad_subst"/>
            <category id="ad_cnt"/>
            <category id="ad_jc"/>
        </categories>
    </category-group>
    <category-group id="bld_group" ugc_valuable="true">
        <label>{{category-groups:bld_group}}</label>
        <categories>
            <category id="bld"/>
            <category id="bld_complex"/>
        </categories>
    </category-group>
</category-groups>'''

    assert define_category_groups_dict == edits_constants.create_category_groups_dict(test_xml_string)


def test_create_action_path(statface_client, job, geobase_fixture):
    job = common.make_count_job(statface_client, job)

    nmaps_edits_raw = [
        Record(
            action=b'object-modified', commit_id='1', region_id='213',
            geom='[[55.733842, 37.588144],[55.733842, 37.588144]]',
            object_category="addr", object_id='1', puid='1',
            fielddate='2020-07-26'
        ),
        Record(
            action=b'object-modified', commit_id='2', region_id='213',
            geom='[[55.733842, 37.588144],[55.733842, 37.588144]]',
            object_category="bld", object_id='2', puid='2',
            fielddate='2020-07-26'
        ),
    ]

    create_action_path = []

    job.local_run(
        sources={
            'nmaps_edits_raw': local.StreamSource(nmaps_edits_raw)
        },
        sinks={
            'create_action_path': local.ListSink(create_action_path)
        }
    )

    assert sorted(create_action_path) == sorted([
        Record(
            action_path='\t_total_\t', commit_id='1', fielddate='2020-07-26',
            object_id='1', puid='1', region_id='213'
        ),
        Record(
            action_path='\t_total_\t', commit_id='2', fielddate='2020-07-26',
            object_id='2', puid='2', region_id='213'
        ),
        Record(
            action_path='\t_total_\tobject-modified\t', commit_id='1',
            fielddate='2020-07-26', object_id='1', puid='1', region_id='213'
        ),
        Record(
            action_path='\t_total_\tobject-modified\t', commit_id='2',
            fielddate='2020-07-26', object_id='2', puid='2', region_id='213'
        ),
    ])


def test_create_object_path(statface_client, job, geobase_fixture):
    job = common.make_count_job(statface_client, job)

    nmaps_edits_raw = [
        Record(
            action='object-modified', commit_id='1', region_id='213',
            geom='[[55.733842, 37.588144],[55.733842, 37.588144]]',
            object_category=b"addr", object_id='1', puid='1',
            fielddate='2020-07-26'
        ),
        Record(
            action='object-modified', commit_id='2', region_id='213',
            geom='[[55.733842, 37.588144],[55.733842, 37.588144]]',
            object_category=b"bld", object_id='2', puid='2',
            fielddate='2020-07-26'
        ),
    ]

    create_object_path = []

    job.local_run(
        sources={
            'nmaps_edits_raw': local.StreamSource(nmaps_edits_raw)
        },
        sinks={
            'create_object_path': local.ListSink(create_object_path)
        }
    )

    assert sorted(create_object_path) == sorted([
        Record(
            commit_id='1', fielddate='2020-07-26', object_id='1',
            object_path='\t_total_\t', puid='1', region_id='213'
        ),
        Record(
            commit_id='1', fielddate='2020-07-26', object_id='1',
            object_path='\t_total_\taddr_group\t', puid='1', region_id='213'
        ),
        Record(
            commit_id='1', fielddate='2020-07-26', object_id='1',
            object_path='\t_total_\taddr_group\taddr\t', puid='1', region_id='213'
        ),
        Record(
            commit_id='2', fielddate='2020-07-26', object_id='2',
            object_path='\t_total_\t', puid='2', region_id='213'
        ),
        Record(
            commit_id='2', fielddate='2020-07-26', object_id='2',
            object_path='\t_total_\tbld_group\t', puid='2', region_id='213'
        ),
        Record(
            commit_id='2', fielddate='2020-07-26', object_id='2',
            object_path='\t_total_\tbld_group\tbld\t', puid='2', region_id='213'
        ),
    ])


def test_create_status_path(statface_client, job, geobase_fixture):
    job = common.make_count_job(statface_client, job)

    edits_count_per_user = [
        Record(date='2020-07-26', edits=100, puid='1'),
        Record(date='2020-07-26', edits=15, puid='2')
    ]
    nmaps_user_raw = [
        Record(puid='1', moderation_status=b'moderator'),
        Record(puid='2', moderation_status=b'cartographer'),
    ]

    create_status_path = []

    job.local_run(
        sources={
            'edits_count_per_user': local.StreamSource(edits_count_per_user),
            'nmaps_user_raw': local.StreamSource(nmaps_user_raw)
        },
        sinks={
            'create_status_path': local.ListSink(create_status_path)
        }
    )

    assert sorted(create_status_path) == sorted([
        Record(
            puid='2', status_path='\t_total_\t'
        ),
        Record(
            puid='2', status_path='\t_total_\tyandex\t'
        ),
        Record(
            puid='2', status_path='\t_total_\tyandex\tnew\t'
        ),
        Record(
            puid='1', status_path='\t_total_\t'
        ),
        Record(
            puid='1', status_path='\t_total_\tusers\t'
        ),
        Record(
            puid='1', status_path='\t_total_\tusers\tmoderator\t'
        )
    ])


def test_create_nmaps_edits(statface_client, job, geobase_fixture):
    job = common.make_count_job(statface_client, job)

    create_action_path = [
        Record(
            action_path='\t_total_\t', commit_id='1', fielddate='2020-07-26',
            object_id='1', puid='1', region_id='213'
        ),
        Record(
            action_path='\t_total_\tobject-modified\t', commit_id='1',
            fielddate='2020-07-26', object_id='1', puid='1', region_id='213'
        ),
    ]

    create_object_path = [
        Record(
            commit_id='1', fielddate='2020-07-26', object_id='1',
            object_path='\t_total_\t', puid='1', region_id='213'
        ),
        Record(
            commit_id='1', fielddate='2020-07-26', object_id='1',
            object_path='\t_total_\taddr_group\t', puid='1', region_id='213'
        ),
        Record(
            commit_id='1', fielddate='2020-07-26', object_id='1',
            object_path='\t_total_\taddr_group\taddr\t', puid='1', region_id='213'
        ),
    ]

    nmaps_edits = []

    job.local_run(
        sources={
            'create_action_path': local.StreamSource(create_action_path),
            'create_object_path': local.StreamSource(create_object_path)
        },
        sinks={
            'nmaps_edits': local.ListSink(nmaps_edits)
        }
    )

    assert sorted(nmaps_edits) == sorted([
        Record(
            action_path='\t_total_\t', commit_id='1', fielddate='2020-07-26',
            object_id='1', object_path='\t_total_\t',
            puid='1', region_id='213'
        ),
        Record(
            action_path='\t_total_\t', commit_id='1', fielddate='2020-07-26',
            object_id='1', object_path='\t_total_\taddr_group\t',
            puid='1', region_id='213'
        ),
        Record(
            action_path='\t_total_\t', commit_id='1', fielddate='2020-07-26',
            object_id='1', object_path='\t_total_\taddr_group\taddr\t',
            puid='1', region_id='213'
        ),
        Record(
            action_path='\t_total_\tobject-modified\t', commit_id='1',
            fielddate='2020-07-26', object_id='1', object_path='\t_total_\t',
            puid='1', region_id='213'
        ),
        Record(
            action_path='\t_total_\tobject-modified\t', commit_id='1',
            fielddate='2020-07-26', object_id='1', object_path='\t_total_\taddr_group\t',
            puid='1', region_id='213'
        ),
        Record(
            action_path='\t_total_\tobject-modified\t', commit_id='1',
            fielddate='2020-07-26', object_id='1', object_path='\t_total_\taddr_group\taddr\t',
            puid='1', region_id='213'
        ),
    ])


def test_make_nmaps_edits_for_count(statface_client, job, geobase_fixture):
    job = common.make_count_job(statface_client, job)

    edits_count_per_user = [
        Record(date='2020-07-26', edits=100, puid='1', fielddate='2020-07-26')
    ]

    nmaps_edits = [
        Record(
            action_path='\t_total_\tobject-modified\t', commit_id='1',
            fielddate='2020-07-26', object_id='1', object_path='\t_total_\t',
            puid='1', region_id='213'
        ),
        Record(
            action_path='\t_total_\tobject-modified\t', commit_id='1',
            fielddate='2020-07-26', object_id='1', object_path='\t_total_\taddr_group\t',
            puid='1', region_id='213'
        ),
        Record(
            action_path='\t_total_\tobject-modified\t', commit_id='1',
            fielddate='2020-07-26', object_id='1', object_path='\t_total_\taddr_group\taddr\t',
            puid='1', region_id='213'
        ),
    ]
    create_status_path = [
        Record(
            fielddate='2020-07-26', moderation_status='moderator', puid='1',
            status_path='\t_total_\t'
        ),
        Record(
            fielddate='2020-07-26', moderation_status='moderator', puid='1',
            status_path='\t_total_\tusers\t'
        ),
        Record(
            fielddate='2020-07-26', moderation_status='moderator', puid='1',
            status_path='\t_total_\tusers\tmoderator\t'
        )
    ]

    make_nmaps_edits_for_count = []

    job.local_run(
        sources={
            'edits_count_per_user': local.StreamSource(edits_count_per_user),
            'nmaps_edits': local.StreamSource(nmaps_edits),
            'create_status_path': local.StreamSource(create_status_path)
        },
        sinks={
            'make_nmaps_edits_for_count': local.ListSink(make_nmaps_edits_for_count)
        }
    )

    assert sorted(make_nmaps_edits_for_count) == sorted([
        Record(
            action_path='\t_total_\tobject-modified\t', commit_id='1',
            fielddate='2020-07-26', moderation_status='moderator',
            object_id='1', object_path='\t_total_\t', puid='1',
            region_id='213', status_path='\t_total_\t'
        ),
        Record(
            action_path='\t_total_\tobject-modified\t', commit_id='1',
            fielddate='2020-07-26', moderation_status='moderator',
            object_id='1', object_path='\t_total_\t', puid='1',
            region_id='213', status_path='\t_total_\tusers\t'
        ),
        Record(
            action_path='\t_total_\tobject-modified\t', commit_id='1',
            fielddate='2020-07-26', moderation_status='moderator',
            object_id='1', object_path='\t_total_\t', puid='1',
            region_id='213', status_path='\t_total_\tusers\tmoderator\t'
        ),
        Record(
            action_path='\t_total_\tobject-modified\t', commit_id='1',
            fielddate='2020-07-26', moderation_status='moderator',
            object_id='1', object_path='\t_total_\taddr_group\t', puid='1',
            region_id='213', status_path='\t_total_\t'
        ),
        Record(
            action_path='\t_total_\tobject-modified\t', commit_id='1',
            fielddate='2020-07-26', moderation_status='moderator',
            object_id='1', object_path='\t_total_\taddr_group\t', puid='1',
            region_id='213', status_path='\t_total_\tusers\t'
        ),
        Record(
            action_path='\t_total_\tobject-modified\t', commit_id='1',
            fielddate='2020-07-26', moderation_status='moderator',
            object_id='1', object_path='\t_total_\taddr_group\t', puid='1',
            region_id='213', status_path='\t_total_\tusers\tmoderator\t'
        ),
        Record(
            action_path='\t_total_\tobject-modified\t', commit_id='1',
            fielddate='2020-07-26', moderation_status='moderator',
            object_id='1', object_path='\t_total_\taddr_group\taddr\t', puid='1',
            region_id='213', status_path='\t_total_\t'
        ),
        Record(
            action_path='\t_total_\tobject-modified\t', commit_id='1',
            fielddate='2020-07-26', moderation_status='moderator',
            object_id='1', object_path='\t_total_\taddr_group\taddr\t', puid='1',
            region_id='213', status_path='\t_total_\tusers\t'
        ),
        Record(
            action_path='\t_total_\tobject-modified\t', commit_id='1',
            fielddate='2020-07-26', moderation_status='moderator',
            object_id='1', object_path='\t_total_\taddr_group\taddr\t', puid='1',
            region_id='213', status_path='\t_total_\tusers\tmoderator\t'
        ),
    ])


def test_prepare_result_counts(statface_client, job, major_regions, geobase_fixture):
    job = common.make_count_job(statface_client, job)

    add_region_tree = [
        Record(
            action_path='\t_total_\tobject-modified\t', commit_id='1',
            fielddate='2020-07-26', moderation_status='moderator',
            object_id='1', object_path='\t_total_\t', puid='1',
            region_id='213', status_path='\t_total_\t', region_tree="\t10000\t"
        ),
        Record(
            action_path='\t_total_\tobject-modified\t', commit_id='1',
            fielddate='2020-07-26', moderation_status='moderator',
            object_id='1', object_path='\t_total_\t', puid='1',
            region_id='213', status_path='\t_total_\t', region_tree="\t10000\t225\t"
        ),
        Record(
            action_path='\t_total_\tobject-modified\t', commit_id='1',
            fielddate='2020-07-26', moderation_status='moderator',
            object_id='1', object_path='\t_total_\t', puid='1',
            region_id='213', status_path='\t_total_\t', region_tree="\t10000\t225\t3\t"
        ),
        Record(
            action_path='\t_total_\tobject-modified\t', commit_id='2',
            fielddate='2020-07-26', moderation_status='moderator',
            object_id='2', object_path='\t_total_\t', puid='2',
            region_id='213', status_path='\t_total_\t', region_tree="\t10000\t"
        ),
        Record(
            action_path='\t_total_\tobject-modified\t', commit_id='2',
            fielddate='2020-07-26', moderation_status='moderator',
            object_id='2', object_path='\t_total_\t', puid='2',
            region_id='213', status_path='\t_total_\t', region_tree="\t10000\t225\t"
        ),
        Record(
            action_path='\t_total_\tobject-modified\t', commit_id='2',
            fielddate='2020-07-26', moderation_status='moderator',
            object_id='2', object_path='\t_total_\t', puid='2',
            region_id='213', status_path='\t_total_\t', region_tree="\t10000\t225\t3\t"
        ),
        Record(
            action_path='\t_total_\tobject-modified\t', commit_id='3',
            fielddate='2020-07-26', moderation_status='moderator',
            object_id='3', object_path='\t_total_\t', puid='2',
            region_id='213', status_path='\t_total_\t', region_tree="\t10000\t"
        ),
        Record(
            action_path='\t_total_\tobject-modified\t', commit_id='3',
            fielddate='2020-07-26', moderation_status='moderator',
            object_id='3', object_path='\t_total_\t', puid='2',
            region_id='213', status_path='\t_total_\t', region_tree="\t10000\t225\t"
        ),
        Record(
            action_path='\t_total_\tobject-modified\t', commit_id='1',
            fielddate='2020-07-26', moderation_status='moderator',
            object_id='1', object_path='\t_total_\t', puid='1',
            region_id='213', status_path='\t_total_\tusers\t', region_tree="\t10000\t"
        ),
        Record(
            action_path='\t_total_\tobject-modified\t', commit_id='1',
            fielddate='2020-07-26', moderation_status='moderator',
            object_id='1', object_path='\t_total_\t', puid='1',
            region_id='213', status_path='\t_total_\tusers\t', region_tree="\t10000\t225\t"
        ),
        Record(
            action_path='\t_total_\tobject-modified\t', commit_id='1',
            fielddate='2020-07-26', moderation_status='moderator',
            object_id='1', object_path='\t_total_\t', puid='1',
            region_id='213', status_path='\t_total_\tusers\t', region_tree="\t10000\t225\t3\t"
        ),
        Record(
            action_path='\t_total_\tobject-modified\t', commit_id='1',
            fielddate='2020-07-26', moderation_status='moderator',
            object_id='1', object_path='\t_total_\t', puid='1',
            region_id='213', status_path='\t_total_\tusers\tmoderator\t', region_tree="\t10000\t"
        ),
        Record(
            action_path='\t_total_\tobject-modified\t', commit_id='1',
            fielddate='2020-07-26', moderation_status='moderator',
            object_id='1', object_path='\t_total_\t', puid='1',
            region_id='213', status_path='\t_total_\tusers\tmoderator\t', region_tree="\t10000\t225\t"
        ),
        Record(
            action_path='\t_total_\tobject-modified\t', commit_id='1',
            fielddate='2020-07-26', moderation_status='moderator',
            object_id='1', object_path='\t_total_\t', puid='1',
            region_id='213', status_path='\t_total_\tusers\tmoderator\t', region_tree="\t10000\t225\t3\t"
        ),
        Record(
            action_path='\t_total_\tobject-modified\t', commit_id='1',
            fielddate='2020-07-26', moderation_status='moderator',
            object_id='1', object_path='\t_total_\taddr_group\t', puid='1',
            region_id='213', status_path='\t_total_\t', region_tree="\t10000\t"
        ),
        Record(
            action_path='\t_total_\tobject-modified\t', commit_id='1',
            fielddate='2020-07-26', moderation_status='moderator',
            object_id='1', object_path='\t_total_\taddr_group\t', puid='1',
            region_id='213', status_path='\t_total_\t', region_tree="\t10000\t225\t"
        ),
        Record(
            action_path='\t_total_\tobject-modified\t', commit_id='1',
            fielddate='2020-07-26', moderation_status='moderator',
            object_id='1', object_path='\t_total_\taddr_group\t', puid='1',
            region_id='213', status_path='\t_total_\t', region_tree="\t10000\t225\t3\t"
        )
    ]

    prepare_result_counts = []

    job.local_run(
        sources={
            'add_region_tree': local.StreamSource(add_region_tree),
            'major_regions': local.StreamSource(major_regions)
        },
        sinks={
            'prepare_result_counts': local.ListSink(prepare_result_counts)
        }
    )

    assert sorted(prepare_result_counts) == sorted([
        Record(
            action_path='\t_total_\tobject-modified\t', count_uniq_commit_id=1,
            count_uniq_object_id=1, count_uniq_puid=1, fielddate='2020-07-26',
            object_path='\t_total_\t', geo_path='\t10000\t',
            status_path='\t_total_\tusers\tmoderator\t'
        ),
        Record(
            action_path='\t_total_\tobject-modified\t', count_uniq_commit_id=1,
            count_uniq_object_id=1, count_uniq_puid=1, fielddate='2020-07-26',
            object_path='\t_total_\t', geo_path='\t10000\t',
            status_path='\t_total_\tusers\t'
        ),
        Record(
            action_path='\t_total_\tobject-modified\t', count_uniq_commit_id=1,
            count_uniq_object_id=1, count_uniq_puid=1, fielddate='2020-07-26',
            object_path='\t_total_\t', geo_path='\t10000\t225\t',
            status_path='\t_total_\tusers\t'
        ),
        Record(
            action_path='\t_total_\tobject-modified\t', count_uniq_commit_id=1,
            count_uniq_object_id=1, count_uniq_puid=1, fielddate='2020-07-26',
            object_path='\t_total_\t', geo_path='\t10000\t225\t',
            status_path='\t_total_\tusers\tmoderator\t'
        ),
        Record(
            action_path='\t_total_\tobject-modified\t', count_uniq_commit_id=1,
            count_uniq_object_id=1, count_uniq_puid=1, fielddate='2020-07-26',
            object_path='\t_total_\t', geo_path='\t10000\t225\t3\t',
            status_path='\t_total_\tusers\t'
        ),
        Record(
            action_path='\t_total_\tobject-modified\t', count_uniq_commit_id=1,
            count_uniq_object_id=1, count_uniq_puid=1, fielddate='2020-07-26',
            object_path='\t_total_\t', geo_path='\t10000\t225\t3\t',
            status_path='\t_total_\tusers\tmoderator\t'
        ),
        Record(
            action_path='\t_total_\tobject-modified\t', count_uniq_commit_id=1,
            count_uniq_object_id=1, count_uniq_puid=1, fielddate='2020-07-26',
            object_path='\t_total_\taddr_group\t', geo_path='\t10000\t',
            status_path='\t_total_\t'
        ),
        Record(
            action_path='\t_total_\tobject-modified\t', count_uniq_commit_id=1,
            count_uniq_object_id=1, count_uniq_puid=1, fielddate='2020-07-26',
            object_path='\t_total_\taddr_group\t', geo_path='\t10000\t225\t',
            status_path='\t_total_\t'
        ),
        Record(
            action_path='\t_total_\tobject-modified\t', count_uniq_commit_id=1,
            count_uniq_object_id=1, count_uniq_puid=1, fielddate='2020-07-26',
            object_path='\t_total_\taddr_group\t', geo_path='\t10000\t225\t3\t',
            status_path='\t_total_\t'
        ),
        Record(
            action_path='\t_total_\tobject-modified\t', count_uniq_commit_id=3,
            count_uniq_object_id=3, count_uniq_puid=2, fielddate='2020-07-26',
            object_path='\t_total_\t', geo_path='\t10000\t',
            status_path='\t_total_\t'
        ),
        Record(
            action_path='\t_total_\tobject-modified\t', count_uniq_commit_id=3,
            count_uniq_object_id=3, count_uniq_puid=2, fielddate='2020-07-26',
            object_path='\t_total_\t', geo_path='\t10000\t225\t',
            status_path='\t_total_\t'
        ),
        Record(
            action_path='\t_total_\tobject-modified\t', count_uniq_commit_id=2,
            count_uniq_object_id=2, count_uniq_puid=2, fielddate='2020-07-26',
            object_path='\t_total_\t', geo_path='\t10000\t225\t3\t',
            status_path='\t_total_\t'
        )
    ])
