from nile.api.v1 import (
    filters as nf,
    aggregators as na,
    extractors as ne,
    datetime as nd,
)
from qb2.api.v1 import (
    extractors as qe,
    filters as qf
)

from .common import (
    MEMORY_LIMIT, COMMON_GROUP_FIELDS, REPORT_DIMENSIONS,
    list_to_tree_str, action_in_window_datetime
)


SECONDS_IN_DAY = 3600.0 * 24


def action_in_window(action_at, window_start, window_end):
    return action_in_window_datetime(
        action_at,
        nd.Datetime.from_iso(window_start),
        nd.Datetime.from_iso(window_end))


def action_in_window_filter(field_name):
    return nf.custom(action_in_window, field_name, 'window_start', 'window_end')


def need_info_on_window_end(need_info_at, un_need_info_at, window_end):
    window_end_datetime = nd.Datetime.from_iso(window_end)

    if (need_info_at is not None and
            nd.Datetime.from_iso(need_info_at) >= window_end_datetime):
        need_info_at = None

    if (un_need_info_at is not None and
            nd.Datetime.from_iso(un_need_info_at) >= window_end_datetime):
        un_need_info_at = None

    return need_info_at is not None and un_need_info_at is None


def resolution_filter(resolution=None, by_robot=None, with_commits=None):
    filters = []
    if resolution is not None:
        filters.append(nf.equals('resolution', resolution))
    if by_robot is not None:
        if by_robot:
            filters.append(nf.equals('mod_status', 'robot'))
        else:
            filters.append(nf.not_(nf.equals('mod_status', 'robot')))
    if with_commits is not None:
        if with_commits:
            filters.append((qf.nonzero('commit_ids')))
        else:
            filters.append(nf.not_(qf.nonzero('commit_ids')))
    filters.append(action_in_window_filter('resolved_at'))
    return nf.and_(*filters)


def reject_reason_filter(reject_reason):
    return nf.and_(
        action_in_window_filter('resolved_at'),
        nf.equals('resolution', 'rejected'),
        nf.equals('reject_reason', reject_reason))


def age_days(created_at, resolved_at, window_end):
    window_end_datetime = nd.Datetime.from_iso(window_end)

    if nd.Datetime.from_iso(created_at) > window_end_datetime:
        return None
    elif resolved_at is None or nd.Datetime.from_iso(resolved_at) > window_end_datetime:
        created_datetime = nd.Datetime.from_iso(created_at)
        return (window_end_datetime - created_datetime).total_seconds() / SECONDS_IN_DAY
    else:
        return None


def created_to_resolved_days(created_at, resolved_at, window_start, window_end):
    if action_in_window(resolved_at, window_start, window_end) and created_at is not None:
        created = nd.Datetime.from_iso(created_at)
        resolved = nd.Datetime.from_iso(resolved_at)
        return (resolved - created).total_seconds() / SECONDS_IN_DAY
    else:
        return None


def calculate_task_metrics(feedback_table):
    return feedback_table.project(
        ne.all(),
        age_days=ne.custom(age_days, 'created_at', 'resolved_at', 'window_end'),
        created_to_resolved_days=ne.custom(
            created_to_resolved_days, 'created_at', 'resolved_at', 'window_start', 'window_end'),

        memory_limit=MEMORY_LIMIT,
    )


def user_path_resolved(resolved_by_login, mod_status, resolved_by_puid):
    if mod_status == 'common':
        return ['_total_', 'users', 'common']
    elif mod_status in ['expert', 'moderator']:
        return ['_total_', 'users', mod_status, '%s (%s)' % (resolved_by_login, resolved_by_puid)]
    elif mod_status in ['cartographer', 'yandex-moderator']:
        return ['_total_', 'yandex', mod_status, '%s (%s)' % (resolved_by_login, resolved_by_puid)]
    elif mod_status == 'robot':
        return ['_total_', 'yandex', mod_status]
    elif mod_status is None:
        return ['_total_', 'None']
    else:
        return ['_total_', mod_status]  # outsource-role, etc.


def calculate_actions_metrics(feedback_table):
    """
    Metrics about actions with feedback inside period
    """
    return feedback_table.groupby(
        *COMMON_GROUP_FIELDS
    ).aggregate(
        created=na.count(action_in_window_filter('created_at')),
        deployed=na.count(action_in_window_filter('deployed_at')),
        need_info=na.count(action_in_window_filter('need_info_at')),
        un_need_info=na.count(action_in_window_filter('un_need_info_at')),

        fbapi_users_created=na.count_distinct(
            'some_user_id', predicate=action_in_window_filter('created_at')),
        fbapi_users_resolved=na.count_distinct(
            'some_user_id', predicate=action_in_window_filter('resolved_at')),
        fbapi_users_accepted=na.count_distinct(
            'some_user_id', predicate=resolution_filter('accepted')),
        fbapi_users_commits=na.count_distinct(
            'some_user_id', predicate=resolution_filter('accepted', with_commits=True)),

        memory_limit=MEMORY_LIMIT,
    )


def calculate_unresolved_metrics(feedback_table):
    """
    Metrics about states of unresolved feedback at the end of the period
    """
    return feedback_table.filter(
        nf.not_(nf.equals('age_days', None))
    ).groupby(
        *COMMON_GROUP_FIELDS
    ).aggregate(
        percentile_95_task_age=na.percentile('age_days', 95),
        unresolved=na.count(),  # 'age_days' is not None only if not resolved before the end of the period
        older_3_days=na.count(qf.compare('age_days', '>', 3)),
        max_task_age=na.max('age_days'),
        unresolved_need_info=na.count(nf.custom(need_info_on_window_end)),

        memory_limit=MEMORY_LIMIT,
    )


def calculate_resolution_metrics(feedback_table):
    """
    Metrics associated with the resolution of feedback
    """
    return feedback_table.project(
        ne.all(),
        qe.custom(
            'user_resolved_path_list',
            user_path_resolved,
            'resolved_by_login',
            'mod_status',
            'resolved_by_puid'
        ).allow_null_dependency().hide(),
        qe.unfold_prefixes('user_resolved_path_prefix', 'user_resolved_path_list').hide(),
        user_resolved_path=ne.custom(list_to_tree_str, 'user_resolved_path_prefix'),

        memory_limit=MEMORY_LIMIT,
    ).groupby(
        *REPORT_DIMENSIONS
    ).aggregate(
        resolved=na.count(action_in_window_filter('resolved_at')),
        count_unique_resolved_by=na.count_distinct(
            'resolved_by_puid',
            predicate=action_in_window_filter('resolved_at')),
        resolved_with_commit=na.count(resolution_filter(resolution='accepted', with_commits=True)),
        percentile_85_created_to_resolved=na.percentile('created_to_resolved_days', 85),
        resolved_accepted=na.count(resolution_filter(resolution='accepted', by_robot=False)),
        resolved_accepted_robot=na.count(resolution_filter(resolution='accepted', by_robot=True)),
        resolved_accepted_total=na.count(resolution_filter(resolution='accepted')),
        resolved_rejected=na.count(resolution_filter(resolution='rejected')),

        reject_no_reason=na.count(nf.or_(reject_reason_filter(''), reject_reason_filter(None))),
        reject_reason_spam=na.count(reject_reason_filter('spam')),
        reject_reason_no_data=na.count(reject_reason_filter('no-data')),
        reject_reason_no_info=na.count(reject_reason_filter('no-info')),
        reject_reason_no_process=na.count(reject_reason_filter('no-process')),
        reject_reason_redirect_to_sprav=na.count(reject_reason_filter('redirect-to-sprav')),
        reject_reason_redirect_to_support=na.count(reject_reason_filter('redirect-to-support')),
        reject_reason_prohibited_by_rules=na.count(reject_reason_filter('prohibited-by-rules')),
        reject_reason_incorrect_data=na.count(reject_reason_filter('incorrect-data')),

        organizations_edit=na.count_distinct('organization_id', resolution_filter(resolution='accepted')),
        organizations_add=na.count(
            nf.and_(
                nf.equals('add_organization', True),
                resolution_filter(resolution='accepted'))),

        memory_limit=MEMORY_LIMIT,
    )


def construct_report_metrics(actions_metrics, unresolved_metrics, resolution_metrics):
    return actions_metrics.join(
        unresolved_metrics,
        by=COMMON_GROUP_FIELDS,
        type='full'  # For a case when there is no actions or unresolved feedback for some folds
    ).project(
        ne.all(),
        user_resolved_path=ne.const('\t_total_\t')
    ).join(
        resolution_metrics,
        by=REPORT_DIMENSIONS,
        type='full'
    )


def calculate_feedback_processing_metrics(feedback_table_unfolded):
    actions_metrics = feedback_table_unfolded.call(calculate_actions_metrics) \
        .label('actions_metrics')
    unresolved_metrics = feedback_table_unfolded.call(calculate_unresolved_metrics) \
        .label('unresolved_metrics')
    resolution_metrics = feedback_table_unfolded.call(calculate_resolution_metrics) \
        .label('resolution_metrics')

    report_metrics = actions_metrics.call(
        construct_report_metrics,
        unresolved_metrics,
        resolution_metrics
    ).label('report_metrics')

    return report_metrics
