# -*- coding: utf-8 -*-

from library.python import resource
from sqlalchemy import create_engine
import xml.etree.ElementTree as ET
import yt.wrapper as yt
import argparse
import logging

YT_ACCOUNT_PATH = "//home/maps/core/nmaps"
YT_MAILING_DIR = "mailing"

logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s %(message)s'
)
logger = logging.getLogger("main-logger")


def environment_name(is_production):
    return 'production' if is_production else 'testing'


class SocialDbConfig:
    def __init__(self, is_production):
        resource_key = 'services_config_' + environment_name(is_production)
        services_config = ET.fromstring(resource.find(resource_key))

        db_config = services_config.find("./common/databases/*[@id='social']")
        db_config_read_element = db_config.find('read')

        self.name = db_config.get('name')
        self.host = db_config_read_element.get('host')
        self.port = db_config_read_element.get('port')
        self.user = db_config_read_element.get('user')


def main():

    parser = argparse.ArgumentParser()

    parser.add_argument(
        '--prod',
        help='Export production profiles',
        action='store_true'
    )
    parser.add_argument(
        '--db-password',
        help='Social database password',
        required=True
    )
    parser.add_argument(
        '--yt-token',
        help='YT authorization token',
        required=True
    )

    args = parser.parse_args()

    logger.info("Initiliazing social database connection")

    db_config = SocialDbConfig(is_production=args.prod)
    db_connection_string = "postgresql://{}:{}@{}:{}/{}"\
        .format(
            db_config.user,
            args.db_password,
            db_config.host,
            db_config.port,
            db_config.name
        )
    db = create_engine(db_connection_string)

    logger.info("Executing social database query")

    query = """

        SELECT
            uid AS yandex_puid,
            email,
            coalesce(news_subscription_modified_at, now())::text AS agreed_at
        FROM social.profile
        WHERE news_subscription = true

    """

    result_set = db.execute(query)
    records = map(lambda row: dict(row.items()), result_set)
    logger.info("Result set rows number: {}".format(result_set.rowcount))

    logger.info("Writing profiles to YT table")

    yt_client = yt.YtClient("hahn", token=args.yt_token)
    yt_result_table = YT_ACCOUNT_PATH + '/'\
        + environment_name(args.prod) + '/'\
        + YT_MAILING_DIR + '/news_subscribers'
    yt_client.write_table(yt_result_table, records)


if __name__ == '__main__':
    main()
