#!/usr/bin/env python3

import argparse
import json
import urllib.request


SCALES = {
    'continual': 's',
    'minutely':  'i',
    'hourly':    'h',
    'daily':     'd',
    'weekly':    'w',
    'monthly':   'm',
    'quarterly': 'q',
    'yearly':    'y',
}


HOST = {
    'production': 'https://upload.stat.yandex-team.ru',
    'beta': 'https://upload.stat-beta.yandex-team.ru',
}


def parse_arguments():
    parser = argparse.ArgumentParser(
        description='Removes data from a Stat report. '
        'Removed data is moved to a kind of a "recycle bin" and keeps there for two days. '
        'If removed data must be restored then Stat team must be communicated as fast as possible. '
        '(see: https://wiki.yandex-team.ru/statbox/Statface/externalreports/#udaleniedannyxizotchetov)'
    )
    parser.add_argument(
        '--env',
        help='Yandex.Stat version to use: %(choices)s (default: %(default)s).',
        choices=HOST.keys(),
        default='beta',
        metavar='ENV'
    )
    parser.add_argument(
        '--token',
        help='Token (can be obtained from https://oauth.yandex-team.ru/authorize?response_type=token&client_id=801af94c94e040848ebe206086a7a4e2).',
        required=True
    )
    parser.add_argument(
        '--report',
        help='Report, data should be removed from.',
        required=True
    )
    parser.add_argument(
        '--date-min',
        help='Date (inclusive, ISO format), data should be removed starting from.',
        required=True
    )
    parser.add_argument(
        '--date-max',
        help='Date (inclusive, ISO format), data should be removed upto.',
        required=True
    )
    parser.add_argument(
        '--scale',
        help='Scale, data should be removed from.',
        choices=SCALES.keys(),
        required=True,
        metavar='SCALE'
    )
    parser.add_argument(
        '--max-distance',
        help='Maximum level of removed trees (default: %(default)i)',
        default=9
    )
    parser.add_argument(
        '--execute',
        help='Must be set to perform deletion (default: %(default)s).',
        action='store_true'
    )

    return parser.parse_args()


def delete_data(environment, token, report_path, scale, min_date, max_date, max_distance, execute):
    url = (
        HOST[environment] + '/_api/report/delete_data/{report_path}?scale={scale}'
        '&date_min={date_min}'
        '&date_max={date_max}'
        '&max_distance={max_distance}'
        '&_allow_default_dimensions=0'  # Remove the whole period (see: https://wiki.yandex-team.ru/statbox/Statface/externalreports/#parametr)
    )
    url = url.format(
        report_path=report_path,
        scale=scale,
        date_min=min_date,
        date_max=max_date,
        max_distance=max_distance
    )
    if execute:
        url += '&execute=1'

    request = urllib.request.Request(
        url,
        headers={'Authorization': 'OAuth {}'.format(token)},
        method='POST'
    )
    response = urllib.request.urlopen(request)
    return json.loads(response.read())


def main():
    args = parse_arguments()
    result = delete_data(
        args.env,
        args.token,
        args.report,
        SCALES[args.scale],
        args.date_min,
        args.date_max,
        args.max_distance,
        args.execute
    )
    print(json.dumps(result, sort_keys=True, indent=4))
