from tvm2.protocol import BlackboxClientId
from tvm2 import TVM2

import json
from urllib import error, request

# Note. OEBS Handlers description can be found in document
# 'YA.MD050.HR.ALL.0522 Функции для ручек OEBSAPI.docx',
# see https://st.yandex-team.ru/OEBSAPPR-1762


OEBS_ENV_PROD = 'production'
OEBS_ENV_TEST = 'testing'

_ENV_TO_OEBS_API_URL = {
    OEBS_ENV_PROD: 'https://oebsapi.mba.yandex-team.ru/rest/',
    OEBS_ENV_TEST: 'https://oebsapi-test.mba.yandex-team.ru/rest/'
}

_ENV_TO_OEBS_TVM_DEST = {
    OEBS_ENV_PROD: 2014842,
    OEBS_ENV_TEST: 2014840,
}


def _get_service_ticket(env, tvm_client_id, tvm_secret):
    assert env in _ENV_TO_OEBS_TVM_DEST, f"Unknown environment '{env}'."

    return TVM2(
        client_id=tvm_client_id,
        secret=tvm_secret,
        blackbox_client=BlackboxClientId.Prod,
        destinations=(_ENV_TO_OEBS_TVM_DEST[env], )
    ).get_service_ticket(
        _ENV_TO_OEBS_TVM_DEST[env]
    )


def _oebs_api(env, tvm_service_ticket, handle, data):
    assert env in _ENV_TO_OEBS_API_URL, f"Unknown environment '{env}'."

    req = request.Request(
        url=_ENV_TO_OEBS_API_URL[env] + handle,
        data=json.dumps(data).encode(),
        headers={
            'Content-Type': 'application/json',
            'X-Ya-Service-Ticket': tvm_service_ticket
        },
        method='POST'
    )

    try:
        result = request.urlopen(req)
    except error.HTTPError as e:
        result = e

    return json.loads(result.read())


def holiday(env, tvm_client_id, tvm_secret, logins, date_from, date_to):
    '''
    Returns official Russian and regional holidays for every login in `logins`
    in interval [date_from; date_to].

    The return value is a json with XXYA_OEBSAPI_PKG.GET_HOLIDAY handler answer.

    Warning! There is a known issue (OEBSSUPPORT-58635): if date_from == date_to
    then no holidays are returned.
    '''
    return _oebs_api(
        env,
        _get_service_ticket(env, tvm_client_id, tvm_secret),
        'holiday',
        {
            'login': logins,
            'date_from': date_from.isoformat(),
            'date_to': date_to.isoformat()
        }
    )


def primary_assignments(env, tvm_client_id, tvm_secret, logins, date):
    '''
    Returns primary assignments for every login in `logins`.

    The return value is a json with XXHR_OEBSAPI_PKG.GET_PRIMARY_ASSG handler
    answer.
    '''
    return _oebs_api(
        env,
        _get_service_ticket(env, tvm_client_id, tvm_secret),
        'primaryAssg',
        {
            'login': logins,
            'date': date.isoformat()
        }
    )
