import re

PIECEWORK_DEPARTMENTS = {
    'outstaff_3676_dep98105',  # Служба картографического производства и геоинформационного контента (Поддержка бизнеса)
    'outstaff_3676_dep52299',  # Отдел картографического производства (Поддержка бизнеса)

    'ext_5873_dep46363',  # Региональные обозреватели
    'ext_5873_dep71395',  # Турция
}

OUTSOURCE_MANAGER_DEPARTMENTS = {
    'ext_outprojects_cart',  # Внешние консультанты Геосервисов / Картографы
    'ext_searchportal_geoserv_transnavi',  # Внешние консультанты Геосервисов / Transnavi.com
}

RE_OUTSOURCE_GROUP = r'outsource-group\.(?P<company>[a-z0-9-]+)'

PARTNER_ACL_ROLES_WITH_COMPANY_NAME = {
    'partner-pedestrian-onfoot_company1',
    'partner-pedestrian-onfoot_company2',
    'partner-pedestrian-onfoot_company3',
    'partner-pedestrian-onfoot_company4',
    'partner-pedestrian-onfoot_company5',
}

PARTNER_ACL_ROLES = {
    'lavka-pedestrian-onfoot',
    'sprav-pedestrian-onfoot',
} | PARTNER_ACL_ROLES_WITH_COMPANY_NAME

PARTNER_ACL_ROLE_COMPANY_NAME_RE = r'(?P<role>.+)_(?P<company>.+)'

PARTNER_DEFAULT_COMPANY_NAME = 'no company'

ROLES = {
    'all':             {'common', 'moderator', 'expert', 'yandex-moderator', 'cartographer', 'outsource-role'},
    'user_anonymized': {'common', 'moderator', 'expert'},
}

TREE_PATH = {
    'user':            ['all', 'user'],
    'robot':           ['all', 'robot'],
    'outsourcer':      ['all', 'piecework', 'outsource'],
    'staff_piecework': ['all', 'piecework', 'staff'],
    'partner':         ['all', 'piecework', 'partner'],
    'staff_salary':    ['all', 'salary'],
}


def outsource_company(outsource_group):
    if outsource_group is None:
        return None

    outsource_group = outsource_group.decode()
    match = re.fullmatch(RE_OUTSOURCE_GROUP, outsource_group)
    if match is not None:
        return match.group('company').encode()

    return b'unknown'


def _is_outsource_manager(departments_urls):
    if departments_urls is None:
        return False
    return bool(set(departments_urls) & OUTSOURCE_MANAGER_DEPARTMENTS)


def _get_outsourcer(puid, yalogin, outsource_company, is_manager):
    if is_manager:
        yalogin = '[manager] ' + yalogin
    return {
        'puid': puid,
        'puid_tree': TREE_PATH['outsourcer'] + [outsource_company, f'{yalogin} ({puid})'],
        'payment': 'piecework',
        'involvement': 'outsource',
        'group': outsource_company,
        'person': yalogin,
    }


def _get_robot(puid, yalogin):
    return {
        'puid': puid,
        'puid_tree': TREE_PATH['robot'] + [f'{yalogin} ({puid})'],
        'payment': 'free',
        'involvement': 'robot',
        'group': 'robot',
        'person': yalogin,
    }


def _get_staff(puid, yalogin, login, last_name, first_name, primary_department, departments_urls):
    path_suffix = [
        primary_department,
        f'{last_name} {first_name} ({login})',
        f'{yalogin} ({puid})',
    ]

    if set(departments_urls) & PIECEWORK_DEPARTMENTS:
        puid_tree = TREE_PATH['staff_piecework'] + path_suffix
        payment = 'piecework'
    else:
        puid_tree = TREE_PATH['staff_salary'] + path_suffix
        payment = 'salary'

    return {
        'puid': puid,
        'puid_tree': puid_tree,
        'payment': payment,
        'involvement': 'staff',
        'group': primary_department,
        'person': path_suffix[1],
    }


def _get_user(puid, yalogin, role):
    if role in ROLES['user_anonymized']:
        puid_tree = TREE_PATH['user'] + [role]
        person = role
    else:
        puid_tree = TREE_PATH['user'] + [role, f'{yalogin} ({puid})']
        person = yalogin

    if role == 'common':
        return {
            'puid': puid,
            'puid_tree': puid_tree
        }

    return {
        'puid': puid,
        'puid_tree': puid_tree,
        'payment': 'free',
        'involvement': 'user',
        'group': role,
        'person': person,
    }


def is_partner(role):
    return role.decode() in PARTNER_ACL_ROLES


def _get_partner(puid, yalogin, partner_acl_role):
    company = PARTNER_DEFAULT_COMPANY_NAME

    match = re.fullmatch(PARTNER_ACL_ROLE_COMPANY_NAME_RE, partner_acl_role)
    if match:
        partner_acl_role = match.group('role')
        company = match.group('company')

    return {
        'puid': puid,
        'puid_tree': TREE_PATH['partner'] + [partner_acl_role, company, yalogin],
        'payment': 'piecework',
        'involvement': 'partner',
        'group': 'sprav' if partner_acl_role.startswith('sprav') else company,
        'person': yalogin,
    }


def get_person(puid, yalogin, role, outsource_company, partner_acl_role, login, last_name, first_name, primary_department, departments_urls):
    is_outsource_manager = _is_outsource_manager(departments_urls)
    if outsource_company is None and is_outsource_manager:
        raise RuntimeError(f'{yalogin} ({puid}) from outsource department "{primary_department}" has no outsource company')

    if outsource_company is not None:
        return _get_outsourcer(puid, yalogin, outsource_company, is_outsource_manager)
    if role == 'robot':
        return _get_robot(puid, yalogin)
    if partner_acl_role is not None:
        return _get_partner(puid, yalogin, partner_acl_role)
    if departments_urls is not None:
        return _get_staff(puid, yalogin, login, last_name, first_name, primary_department, departments_urls)
    return _get_user(puid, yalogin, role)
