from maps.wikimap.stat.tasks_payment.dictionaries.task_tariff_map.lib import task_tariff_map

from maps.wikimap.stat.tasks_payment.dictionaries.tariff.schema import TariffsSchema

from library.python import resource
from nile.api.v1.clusters.yt import Hahn
from textwrap import dedent
from yt.wrapper.ypath import ypath_join
import argparse
import datetime
import yaml


TARIFF_TYPES = ['cartographic', 'geotest']

TARIFF_TYPE_TO_DIR = {
    'cartographic': '//home/maps/core/nmaps/analytics/tasks_payment/dictionaries/task_tariff_map',
    'geotest':      '//home/maps-qa/analytics/tasks_payment/dictionaries/task_tariff_map',
}


def parse_args():
    parser = argparse.ArgumentParser(
        description=dedent('''\
            Calculates `task_tariff_map` dictionary.

            Note. The generated dictionary is stored on the Hahn YT cluster. The token to
            connect to the cluster must be provided either by means of YT_TOKEN environment
            variable or `~/.yt/token` file.
        '''),
        formatter_class=argparse.RawTextHelpFormatter
    )
    parser.add_argument(
        '--tariff-type',
        help=dedent('''\
            Tariff type to generate dictionary for.
        '''),
        required=True,
        choices=TARIFF_TYPES
    )
    parser.add_argument(
        '--out-yt-dir',
        help=dedent('''\
            YT directory to write the generated tables to.
            Warning! Existing tables for specified date will be rewritten.
            Default directories depends on choosen tariff type:
        ''') + '\n'.join(
            f'- {tariff_type}: {yt_dir}' for tariff_type, yt_dir in TARIFF_TYPE_TO_DIR.items()
        )
    )
    parser.add_argument(
        '--date',
        help='The date (YYYY-MM-DD) tariffs must be calculated for. (default: today)',
        type=datetime.date.fromisoformat,
        default=datetime.date.today()
    )

    result = parser.parse_args()

    if result.out_yt_dir is None:
        result.out_yt_dir = TARIFF_TYPE_TO_DIR[result.tariff_type]

    return result


def make_records(tariff_type, date):
    result = []
    for tariffs_description_yaml in resource.itervalues(f'maps/tasks_tariffs/{tariff_type}/'):
        result += task_tariff_map.make_records(
            TariffsSchema().load(
                yaml.full_load(
                    tariffs_description_yaml.decode()
                )
            ),
            date
        )
    return result


def main():
    args = parse_args()

    Hahn().write(
        ypath_join(args.out_yt_dir, args.date.isoformat()),
        make_records(args.tariff_type, args.date)
    )
