from maps.wikimap.stat.tasks_payment.reports.cartographic_report.lib import report
from maps.wikimap.stat.tasks_payment.reports.cartographic_report.lib.report import (
    _DEFAULT_REGION_TREE,
    _DEFAULT_SCHEDULE,
    _FULL_ACCESS_GROUP,
    _HOLIDAY_PAY_RATE_FACTOR,
    _TASK_ID_CURRENT_EMPLOYEE
)

from maps.wikimap.stat.tasks_payment.dictionaries.tariff.schema import TARIFF_RUB_PER_SEC
from maps.wikimap.stat.libs.common.lib.geobase_region import EARTH_REGION_ID

from maps.wikimap.stat.libs import nile_ut

from nile.api.v1 import Record
from nile.drivers.common.progress import CommandFailedError

import pytest


def geobase_region_id_mock(lon, lat):
    return round(lat + lon)


@pytest.fixture
def geobase_fixture(monkeypatch):
    monkeypatch.setattr(
        'maps.wikimap.stat.tasks_payment.reports.cartographic_report.lib.report.GEOBASE_FILES',
        []
    )
    monkeypatch.setattr(
        'maps.wikimap.stat.tasks_payment.reports.cartographic_report.lib.report.geobase_region_id',
        geobase_region_id_mock
    )


def test_should_check_is_employee_current():
    assert report._is_employee_current(date=b'2020-04-27', quit_at=b'2020-04-26') is False
    assert report._is_employee_current(date=b'2020-04-27', quit_at=b'2020-04-27') is False
    assert report._is_employee_current(date=b'2020-04-27', quit_at=b'2020-04-28') is True
    assert report._is_employee_current(date=b'2020-04-27', quit_at=None) is True


def test_should_make_current_employees_log():
    result = nile_ut.yt_run(
        report._make_current_employees_log,
        date=b'2020-04-27',
        staff_dump=nile_ut.Table([
            Record(login=b'staff_login_1', quit_at=None),
            Record(login=b'staff_login_2', quit_at=None),
            Record(login=b'staff_login_3', quit_at=None),
            Record(login=b'staff_login_4', quit_at=b'2020-03-31'),
        ]),
        linked_accounts=nile_ut.Table([
            Record(staff_login=b'staff_login_1', puid=1, is_primary_link=True),
            Record(staff_login=b'staff_login_3', puid=3, is_primary_link=True),
            Record(staff_login=b'staff_login_4', puid=4, is_primary_link=True),
        ])
    )

    assert sorted([
        Record(iso_datetime=b'2020-04-27', puid=1, task_id=_TASK_ID_CURRENT_EMPLOYEE, mrc_region=None, quantity=0.0,
               geom=None, lat_min=None, lon_min=None, lat_max=None, lon_max=None),
        Record(iso_datetime=b'2020-04-27', puid=3, task_id=_TASK_ID_CURRENT_EMPLOYEE, mrc_region=None, quantity=0.0,
               geom=None, lat_min=None, lon_min=None, lat_max=None, lon_max=None),
    ]) == sorted(result)


def test_should_use_primary_link_when_make_current_employees_log():
    result = nile_ut.yt_run(
        report._make_current_employees_log,
        date=b'2020-04-27',
        staff_dump=nile_ut.Table([
            Record(login=b'staff_login_1', quit_at=None),
        ]),
        linked_accounts=nile_ut.Table([
            Record(staff_login=b'staff_login_1', puid=11, is_primary_link=False),
            Record(staff_login=b'staff_login_1', puid=12, is_primary_link=True),
            Record(staff_login=b'staff_login_1', puid=13, is_primary_link=False),
        ])
    )

    assert sorted([
        Record(iso_datetime=b'2020-04-27', puid=12, task_id=_TASK_ID_CURRENT_EMPLOYEE, mrc_region=None, quantity=0.0,
               geom=None, lat_min=None, lon_min=None, lat_max=None, lon_max=None),
    ]) == sorted(result)


def test_should_leave_current_employees_only_when_make_current_employees_log():
    result = nile_ut.yt_run(
        report._make_current_employees_log,
        date=b'2020-04-27',
        staff_dump=nile_ut.Table([
            Record(login=b'staff_login_1', quit_at=b'2020-04-27'),
            Record(login=b'staff_login_2', quit_at=None),
            Record(login=b'staff_login_3', quit_at=b'2020-04-26'),
            Record(login=b'staff_login_4', quit_at=b'2020-04-28'),
        ]),
        linked_accounts=nile_ut.Table([
            Record(staff_login=b'staff_login_1', puid=1, is_primary_link=True),
            Record(staff_login=b'staff_login_2', puid=2, is_primary_link=True),
            Record(staff_login=b'staff_login_3', puid=3, is_primary_link=True),
            Record(staff_login=b'staff_login_4', puid=4, is_primary_link=True),
        ])
    )

    assert sorted([
        Record(iso_datetime=b'2020-04-27', puid=2, task_id=_TASK_ID_CURRENT_EMPLOYEE, mrc_region=None, quantity=0.0,
               geom=None, lat_min=None, lon_min=None, lat_max=None, lon_max=None),
        Record(iso_datetime=b'2020-04-27', puid=4, task_id=_TASK_ID_CURRENT_EMPLOYEE, mrc_region=None, quantity=0.0,
               geom=None, lat_min=None, lon_min=None, lat_max=None, lon_max=None),
    ]) == sorted(result)


def test_should_concat_logs_and_filter_out_geom_column():
    result = nile_ut.yt_run(
        report._concat_logs,
        nile_ut.Job(),
        b'2020-04-27',
        nile_ut.Table([
            Record(iso_datetime=b'2020-04-27 01:01:01+01:01', puid=1, task_id=b'1', quantity=1.0, geom=b'geom 1', lat_min=1.1, lon_min=1.1, lat_max=1.1, lon_max=1.1),
            Record(iso_datetime=b'2020-04-27 02:02:02+02:02', puid=2, task_id=b'2', quantity=2.0, geom=b'geom 2', lat_min=2.2, lon_min=2.2, lat_max=2.2, lon_max=2.2),
        ]),
        nile_ut.Table([
            Record(iso_datetime=b'2020-04-27 03:03:03+03:03', puid=3, task_id=b'3', quantity=3.0, geom=b'geom 3', lat_min=3.3, lon_min=3.3, lat_max=3.3, lon_max=3.3)
        ])
    )

    assert sorted([
        Record(fielddate=b'2020-04-27', puid=1, task_id=b'1', quantity=1.0, lat_min=1.1, lon_min=1.1, lat_max=1.1, lon_max=1.1),
        Record(fielddate=b'2020-04-27', puid=2, task_id=b'2', quantity=2.0, lat_min=2.2, lon_min=2.2, lat_max=2.2, lon_max=2.2),
        Record(fielddate=b'2020-04-27', puid=3, task_id=b'3', quantity=3.0, lat_min=3.3, lon_min=3.3, lat_max=3.3, lon_max=3.3)
    ]) == sorted(result)


def test_should_convert_bbox_to_region_id(geobase_fixture):
    result = nile_ut.yt_run(
        report._bbox_to_region_id,
        log=nile_ut.Table([
            Record(fielddate=b'2020-04-27', puid=1, task_id=b'1', quantity=1.0, lat_min=1.0, lon_min=2.0, lat_max=3.0, lon_max=4.0)
        ])
    )

    assert sorted(
        [Record(fielddate=b'2020-04-27', puid=1, task_id=b'1', quantity=1.0, region_id=5)]
    ) == sorted(result)


def test_should_use_earth_region_id_if_no_bbox(geobase_fixture):
    result = nile_ut.yt_run(
        report._bbox_to_region_id,
        log=nile_ut.Table([
            Record(fielddate=b'2020-04-27', puid=1, task_id=b'1', quantity=1.0, lat_min=None, lon_min=1.0,  lat_max=1.0,  lon_max=1.0),
            Record(fielddate=b'2020-04-27', puid=2, task_id=b'2', quantity=2.0, lat_min=2.0,  lon_min=None, lat_max=2.0,  lon_max=2.0),
            Record(fielddate=b'2020-04-27', puid=3, task_id=b'3', quantity=3.0, lat_min=3.0,  lon_min=3.0,  lat_max=None, lon_max=3.0),
            Record(fielddate=b'2020-04-27', puid=4, task_id=b'4', quantity=4.0, lat_min=4.0,  lon_min=4.0,  lat_max=4.0,  lon_max=None)
        ])
    )

    assert sorted([
        Record(fielddate=b'2020-04-27', puid=1, task_id=b'1', quantity=1.0, region_id=EARTH_REGION_ID),
        Record(fielddate=b'2020-04-27', puid=2, task_id=b'2', quantity=2.0, region_id=EARTH_REGION_ID),
        Record(fielddate=b'2020-04-27', puid=3, task_id=b'3', quantity=3.0, region_id=EARTH_REGION_ID),
        Record(fielddate=b'2020-04-27', puid=4, task_id=b'4', quantity=4.0, region_id=EARTH_REGION_ID)
    ]) == sorted(result)


def test_should_add_staff_login():
    result = nile_ut.yt_run(
        report._add_staff_login,
        log=nile_ut.Table([
            Record(fielddate=b'2020-04-27', puid=1, task_id=b'11', quantity=1.1, region_id=EARTH_REGION_ID, mrc_region=b'region 11'),
            Record(fielddate=b'2020-04-27', puid=1, task_id=b'12', quantity=1.2, region_id=EARTH_REGION_ID, mrc_region=b'region 12'),
            Record(fielddate=b'2020-04-27', puid=2, task_id=b'21', quantity=2.1, region_id=EARTH_REGION_ID, mrc_region=b'region 21'),
            Record(fielddate=b'2020-04-27', puid=3, task_id=b'31', quantity=3.1, region_id=EARTH_REGION_ID, mrc_region=b'region 31'),
        ]),
        linked_accounts=nile_ut.Table([
            Record(puid=1, login=b'nmaps login 1', staff_uid=101, staff_login=b'staff_login_1'),
            Record(puid=2, login=b'nmaps login 2', staff_uid=102, staff_login=b'staff_login_2'),
        ])
    )

    assert sorted([
        Record(fielddate=b'2020-04-27', puid=1, task_id=b'11', mrc_region=b'region 11', quantity=1.1, region_id=EARTH_REGION_ID, staff_login=b'staff_login_1'),
        Record(fielddate=b'2020-04-27', puid=1, task_id=b'12', mrc_region=b'region 12', quantity=1.2, region_id=EARTH_REGION_ID, staff_login=b'staff_login_1'),
        Record(fielddate=b'2020-04-27', puid=2, task_id=b'21', mrc_region=b'region 21', quantity=2.1, region_id=EARTH_REGION_ID, staff_login=b'staff_login_2'),
        Record(fielddate=b'2020-04-27', puid=3, task_id=b'31', mrc_region=b'region 31', quantity=3.1, region_id=EARTH_REGION_ID, staff_login=None),
    ]) == sorted(result)


def test_should_add_user_holiday():
    result = nile_ut.yt_run(
        report._add_user_holidays,
        date=b'2020-04-27',
        log=nile_ut.Table([
            Record(test_column=1, staff_login=b'login 1'),
            Record(test_column=2, staff_login=b'login 2'),
            Record(test_column=3, staff_login=b'login 3'),
        ]),
        user_holidays=nile_ut.Table([
            Record(staff_login=b'login 2', date=b'2020-04-26'),
            Record(staff_login=b'login 2', date=b'2020-04-27'),
            Record(staff_login=b'login 3', date=b'2020-04-28'),
        ])
    )

    assert sorted([
        Record(test_column=1, staff_login=b'login 1', holiday=None),
        Record(test_column=2, staff_login=b'login 2', holiday=True),
        Record(test_column=3, staff_login=b'login 3', holiday=None),
    ]) == sorted(result)


def test_should_add_basic_units():
    result = nile_ut.yt_run(
        report._add_basic_units,
        log=nile_ut.Table([
            Record(staff_login=b'login 1', test_column=1),
            Record(staff_login=b'login 2', test_column=2),
        ]),
        basic_units=nile_ut.Table([
            Record(staff_login=b'login 1', basic_unit_to_rub=11),
            Record(staff_login=b'login 2', basic_unit_to_rub=2.2),
        ])
    )

    assert sorted([
        Record(staff_login=b'login 1', basic_unit_to_rub=11,  test_column=1),
        Record(staff_login=b'login 2', basic_unit_to_rub=2.2, test_column=2),
    ]) == sorted(result)


def test_should_add_work_schedule():
    result = nile_ut.yt_run(
        report._add_work_schedule,
        log=nile_ut.Table([
            Record(staff_login=b'staff_login_1', test_column=1),
            Record(staff_login=b'staff_login_2', test_column=2),
            Record(staff_login=None,             test_column=3),
        ]),
        work_schedule=nile_ut.Table([
            Record(staff_login=b'staff_login_2', schedule=b'schedule 2'),
            Record(staff_login=b'staff_login_4', schedule=b'schedule 4'),
        ])
    )

    assert sorted([
        Record(staff_login=b'staff_login_1', test_column=1, schedule_tree=b'\tall\t'),
        Record(staff_login=b'staff_login_1', test_column=1, schedule_tree=b'\tall\t' + _DEFAULT_SCHEDULE + b'\t'),
        Record(staff_login=b'staff_login_2', test_column=2, schedule_tree=b'\tall\t'),
        Record(staff_login=b'staff_login_2', test_column=2, schedule_tree=b'\tall\tschedule 2\t'),
        Record(staff_login=None,             test_column=3, schedule_tree=b'\tall\t'),
        Record(staff_login=None,             test_column=3, schedule_tree=b'\tall\t' + _DEFAULT_SCHEDULE + b'\t'),
    ]) == sorted(result)


def test_should_convert_region_id_to_region_name():
    result = nile_ut.yt_run(
        report._region_id_to_region_name_tree,
        log=nile_ut.Table([
            Record(region_id=1, log_test_column=1, holiday=True),
            Record(region_id=2, log_test_column=2),
        ]),
        major_regions_map=nile_ut.Table([
            Record(region_id=b'1', region_tree=b'parent region 1/region 1', major_regions_map_test_column=1),
            Record(region_id=b'1', region_tree=b'parent region 1',          major_regions_map_test_column=2),
            Record(region_id=b'2', region_tree=b'parent region 2',          major_regions_map_test_column=3),
        ])
    )

    assert sorted([
        Record(region_name_tree=b'parent region 1/region 1', log_test_column=1, holiday=True),
        Record(region_name_tree=b'parent region 1',          log_test_column=1, holiday=True),
        Record(region_name_tree=b'parent region 2',          log_test_column=2),
    ]) == sorted(result)


def test_should_use_default_region_name_for_unknown_region_id():
    result = nile_ut.yt_run(
        report._region_id_to_region_name_tree,
        log=nile_ut.Table([
            Record(fielddate=b'2020-04-27', puid=1, task_id=b'task', quantity=1.0, region_id=1, staff_login=b'staff_login', schedule_tree=b'schedule')
        ]),
        major_regions_map=nile_ut.Table([
            Record(region_id=b'2', region_tree=b'region 2')
        ])
    )

    assert sorted([
        Record(fielddate=b'2020-04-27', puid=1, task_id=b'task', quantity=1.0, region_name_tree=_DEFAULT_REGION_TREE, staff_login=b'staff_login', schedule_tree=b'schedule'),
    ]) == sorted(result)


def test_should_fail_on_absent_tree_values():
    with pytest.raises(RuntimeError, match='No b\'tree_name\' for b\'value\' = \'1\''):
        list(
            report._check_absent_tree_values(tree_name_col=b'tree_name', value_col=b'value')(
                [Record(tree_name=None, value=1)]
            )
        )


def test_should_not_fail_on_valid_tree_values():
    records = [Record(tree_name=b'tree: 1', value=1)]
    records == report._check_absent_tree_values(tree_name_col=b'tree_name', value_col=b'value')(records)


def test_should_add_task_name_time_spent_and_cost():
    result = nile_ut.yt_run(
        report._add_task_name_and_cost,
        log=nile_ut.Table([
            Record(task_id=b'task 1', quantity=1.0, basic_unit_to_rub=1),
            Record(task_id=b'task 2', quantity=2.0, basic_unit_to_rub=2, holiday=None),
            Record(task_id=b'task 1', quantity=3.0, basic_unit_to_rub=3, holiday=True),
        ]),
        task_tariff_map=nile_ut.Table([
            Record(task_id=b'task 1', seconds_per_task=1, task_name_tree=b'\tall\ttariff 1\ttask 1\t'),
            Record(task_id=b'task 1', seconds_per_task=1, task_name_tree=b'\tall\ttariff 1\t'),
            Record(task_id=b'task 1', seconds_per_task=1, task_name_tree=b'\tall\t'),
            Record(task_id=b'task 2', seconds_per_task=2, task_name_tree=b'\tall\ttariff 2\t__MRC_REGION__\ttask 2\t'),
            Record(task_id=b'task 2', seconds_per_task=2, task_name_tree=b'\tall\ttariff 2\t__MRC_REGION__\t'),
            Record(task_id=b'task 2', seconds_per_task=2, task_name_tree=b'\tall\ttariff 2\t'),
            Record(task_id=b'task 2', seconds_per_task=2, task_name_tree=b'\tall'),
        ])
    )

    assert sorted([
        Record(task_name_tree=b'\tall\ttariff 1\ttask 1\t', quantity=1.0, time_spent_sec=1.0, basic_unit_to_rub=1.0,
               cost_rub=1.0 * TARIFF_RUB_PER_SEC,
               cost_basic_unit=1.0 * TARIFF_RUB_PER_SEC / 1.0),
        Record(task_name_tree=b'\tall\ttariff 1\t', quantity=1.0, time_spent_sec=1.0, basic_unit_to_rub=1.0,
               cost_rub=1.0 * TARIFF_RUB_PER_SEC,
               cost_basic_unit=1.0 * TARIFF_RUB_PER_SEC / 1.0),
        Record(task_name_tree=b'\tall\t', quantity=1.0, time_spent_sec=1.0, basic_unit_to_rub=1.0,
               cost_rub=1.0 * TARIFF_RUB_PER_SEC,
               cost_basic_unit=1.0 * TARIFF_RUB_PER_SEC / 1.0),

        Record(task_name_tree=b'\tall\ttariff 2\t__MRC_REGION__\ttask 2\t', quantity=2.0, holiday=None, time_spent_sec=4.0, basic_unit_to_rub=2.0,
               cost_rub=4.0 * TARIFF_RUB_PER_SEC,
               cost_basic_unit=4.0 * TARIFF_RUB_PER_SEC / 2.0),
        Record(task_name_tree=b'\tall\ttariff 2\t__MRC_REGION__\t', quantity=2.0, holiday=None, time_spent_sec=4.0, basic_unit_to_rub=2.0,
               cost_rub=4.0 * TARIFF_RUB_PER_SEC,
               cost_basic_unit=4.0 * TARIFF_RUB_PER_SEC / 2.0),
        Record(task_name_tree=b'\tall\ttariff 2\t', quantity=2.0, holiday=None, time_spent_sec=4.0, basic_unit_to_rub=2.0,
               cost_rub=4.0 * TARIFF_RUB_PER_SEC,
               cost_basic_unit=4.0 * TARIFF_RUB_PER_SEC / 2.0),
        Record(task_name_tree=b'\tall', quantity=2.0, holiday=None, time_spent_sec=4.0, basic_unit_to_rub=2.0,
               cost_rub=4.0 * TARIFF_RUB_PER_SEC,
               cost_basic_unit=4.0 * TARIFF_RUB_PER_SEC / 2.0),

        Record(task_name_tree=b'\tall\ttariff 1\ttask 1\t', quantity=3.0, holiday=True, time_spent_sec=3.0, basic_unit_to_rub=3.0,
               cost_rub=3.0 * TARIFF_RUB_PER_SEC * _HOLIDAY_PAY_RATE_FACTOR,
               cost_basic_unit=3.0 * TARIFF_RUB_PER_SEC / 3.0),
        Record(task_name_tree=b'\tall\ttariff 1\t', quantity=3.0, holiday=True, time_spent_sec=3.0, basic_unit_to_rub=3.0,
               cost_rub=3.0 * TARIFF_RUB_PER_SEC * _HOLIDAY_PAY_RATE_FACTOR,
               cost_basic_unit=3.0 * TARIFF_RUB_PER_SEC / 3.0),
        Record(task_name_tree=b'\tall\t', quantity=3.0, holiday=True, time_spent_sec=3.0, basic_unit_to_rub=3.0,
               cost_rub=3.0 * TARIFF_RUB_PER_SEC * _HOLIDAY_PAY_RATE_FACTOR,
               cost_basic_unit=3.0 * TARIFF_RUB_PER_SEC / 3.0),
    ]) == sorted(result)


def test_should_fail_on_unknown_task_id():
    with pytest.raises(CommandFailedError):
        nile_ut.yt_run(
            report._add_task_name_and_cost,
            log=nile_ut.Table([
                Record(task_id=b'some task', quantity=1.0)
            ]),
            task_tariff_map=nile_ut.Table([
                Record(task_id=b'task 1', task_name_tree=b'\tall\tgroup 1\ttariff 1\ttask 1\t', seconds_per_task=1),
            ])
        )


def test_should_embed_mrc_region_into_task_name_tree():
    result = nile_ut.yt_run(
        report._embed_mrc_region_into_task_name_tree,
        log=nile_ut.Table([
            Record(task_name_tree=b'\tall\tmrc feedback\t__MRC_REGION__\ttariff 1\ttask 1\t', mrc_region=b'mrc region 1', test_column=b'1'),
            Record(task_name_tree=b'\tall\tmrc feedback\t__MRC_REGION__\ttariff 1\t', mrc_region=b'mrc region 1', test_column=b'1'),
            Record(task_name_tree=b'\tall\tmrc feedback\t__MRC_REGION__\t', mrc_region=b'mrc region 1', test_column=b'1'),
            Record(task_name_tree=b'\tall\tmrc feedback\t', mrc_region=b'mrc region 1', test_column=b'1'),
            Record(task_name_tree=b'\tall\t', mrc_region=b'mrc region 1', test_column=b'1'),
        ])
    )

    assert sorted([
        Record(task_name_tree=b'\tall\tmrc feedback\tmrc region 1\ttariff 1\ttask 1\t', test_column=b'1'),
        Record(task_name_tree=b'\tall\tmrc feedback\tmrc region 1\ttariff 1\t', test_column=b'1'),
        Record(task_name_tree=b'\tall\tmrc feedback\tmrc region 1\t', test_column=b'1'),
        Record(task_name_tree=b'\tall\tmrc feedback\t', test_column=b'1'),
        Record(task_name_tree=b'\tall\t', test_column=b'1'),
    ]) == sorted(result)


def test_should_fail_on_task_without_required_mrc_region():
    with pytest.raises(CommandFailedError, match='No mrc_region'):
        nile_ut.yt_run(
            report._embed_mrc_region_into_task_name_tree,
            log=nile_ut.Table([
                Record(task_name_tree=b'\tall\tmrc feedback\t__MRC_REGION__\ttariff1\ttask 1\t', mrc_region=None),
            ])
        )


def test_should_remove_non_aggregatable_data_from_records_without_login():
    result = report._remove_non_aggregatable_data_from_records_without_login([
        Record(puid_tree=b'\tall\t',                   staff_login=b'login_1', basic_unit_to_rub=1.0, holiday=True),
        Record(puid_tree=b'\tall\tuser 1 (login_1)\t', staff_login=b'login_1', basic_unit_to_rub=1.0, holiday=True),
        Record(puid_tree=b'\tall\tuser 2 (login_2)\t',                         basic_unit_to_rub=1.0, holiday=True),
        Record(puid_tree=b'\tall\tuser 3 (login_3)\t', staff_login=None,       basic_unit_to_rub=1.0, holiday=True),
        Record(puid_tree=b'\tall\tuser 4 (login_4)\t', staff_login=b'login_4'),
    ])

    assert sorted([
        Record(puid_tree=b'\tall\t',                   staff_login=b'login_1', basic_unit_to_rub=None, holiday=None),
        Record(puid_tree=b'\tall\tuser 1 (login_1)\t', staff_login=b'login_1', basic_unit_to_rub=1.0,  holiday=True),
        Record(puid_tree=b'\tall\tuser 2 (login_2)\t',                         basic_unit_to_rub=None, holiday=None),
        Record(puid_tree=b'\tall\tuser 3 (login_3)\t', staff_login=None,       basic_unit_to_rub=None, holiday=None),
        Record(puid_tree=b'\tall\tuser 4 (login_4)\t', staff_login=b'login_4'),
    ]) == sorted(result)


def test_should_add_puid_tree():
    result = nile_ut.yt_run(
        report._add_puid_tree,
        log=nile_ut.Table([
            Record(puid=1, quantity=1.0, staff_login=b'login_1', holiday=True,  basic_unit_to_rub=1.0),
            Record(puid=2, quantity=2.0, staff_login=b'login_2', holiday=False, basic_unit_to_rub=2.0)
        ]),
        puid_map=nile_ut.Table([
            Record(puid=1, puid_tree=b'\tall\t'),
            Record(puid=1, puid_tree=b'\tall\tuser 1 (login_1)\t'),
            Record(puid=2, puid_tree=b'\tall\tuser 2 (login_2)\t')
        ])
    )

    assert sorted([
        Record(quantity=1.0, puid_tree=b'\tall\t',                   staff_login=b'login_1', holiday=None,  basic_unit_to_rub=None),
        Record(quantity=1.0, puid_tree=b'\tall\tuser 1 (login_1)\t', staff_login=b'login_1', holiday=True,  basic_unit_to_rub=1.0),
        Record(quantity=2.0, puid_tree=b'\tall\tuser 2 (login_2)\t', staff_login=b'login_2', holiday=False, basic_unit_to_rub=2.0)
    ]) == sorted(result)


def test_should_fail_on_unknown_puid():
    with pytest.raises(CommandFailedError):
        nile_ut.yt_run(
            report._add_puid_tree,
            log=nile_ut.Table([
                Record(puid=1, quantity=1.0, staff_login=b'login_1', holiday=True)
            ]),
            puid_map=nile_ut.Table([
                Record(puid=2, puid_tree=b'\tall\tuser 2 (login_2)\t')
            ])
        )


def test_should_make_acl():
    result = nile_ut.yt_run(
        report._make_acl,
        log=nile_ut.Table([
            Record(fielddate=b'2020-04-27', staff_login=b'staff_login', puid_tree=b'\tall\t'),
            Record(fielddate=b'2020-04-27', staff_login=b'staff_login', puid_tree=b'\tall\tstaff name (staff_login)\t'),
            Record(fielddate=b'2020-04-27', staff_login=b'staff_login', puid_tree=b'\tall\tstaff name (staff_login)\tnmaps login (puid 1)'),
        ]),
    )

    assert sorted([
        Record(fielddate=b'2020-04-27', puid_tree=b'\tall\t', _acl=_FULL_ACCESS_GROUP),
        Record(fielddate=b'2020-04-27', puid_tree=b'\tall\tstaff name (staff_login)\t', _acl=_FULL_ACCESS_GROUP + b',@staff_login'),
        Record(fielddate=b'2020-04-27', puid_tree=b'\tall\tstaff name (staff_login)\tnmaps login (puid 1)', _acl=_FULL_ACCESS_GROUP + b',@staff_login'),
    ]) == sorted(result)


def test_should_not_show_entries_prior_to_acl_start_date():
    result = nile_ut.yt_run(
        report._make_acl,
        log=nile_ut.Table([
            Record(fielddate=b'2020-04-26', staff_login=b'staff_login', puid_tree=b'\tstaff name (staff_login)\t'),
            Record(fielddate=b'2020-04-27', staff_login=b'staff_login', puid_tree=b'\tstaff name (staff_login)\t'),
            Record(fielddate=b'2020-04-28', staff_login=b'staff_login', puid_tree=b'\tstaff name (staff_login)\t'),
        ]),
    )

    assert sorted([
        Record(fielddate=b'2020-04-26', puid_tree=b'\tstaff name (staff_login)\t', _acl=_FULL_ACCESS_GROUP),
        Record(fielddate=b'2020-04-27', puid_tree=b'\tstaff name (staff_login)\t', _acl=_FULL_ACCESS_GROUP + b',@staff_login'),
        Record(fielddate=b'2020-04-28', puid_tree=b'\tstaff name (staff_login)\t', _acl=_FULL_ACCESS_GROUP + b',@staff_login'),
    ]) == sorted(result)


def test_should_not_show_entries_without_staff_login():
    result = nile_ut.yt_run(
        report._make_acl,
        log=nile_ut.Table([
            Record(fielddate=b'2020-04-27', staff_login=b'staff_login_1', puid_tree=b'\tstaff name 1 (staff_login_1)\t'),
            Record(fielddate=b'2020-04-27', staff_login=None,             puid_tree=b'\tstaff name 2 (staff_login_2)\t'),
        ]),
    )

    assert sorted([
        Record(fielddate=b'2020-04-27', puid_tree=b'\tstaff name 1 (staff_login_1)\t', _acl=_FULL_ACCESS_GROUP + b',@staff_login_1'),
        Record(fielddate=b'2020-04-27', puid_tree=b'\tstaff name 2 (staff_login_2)\t', _acl=_FULL_ACCESS_GROUP),
    ]) == sorted(result)


def test_should_not_show_entries_if_puid_tree_has_wrong_format():
    result = nile_ut.yt_run(
        report._make_acl,
        log=nile_ut.Table([
            Record(fielddate=b'2020-04-27', staff_login=b'staff_login_1', puid_tree=b'\tstaff name 1 staff_login_1\t'),
            Record(fielddate=b'2020-04-27', staff_login=b'staff_login_2', puid_tree=b'\tstaff name 2 (staff_login_2)\t'),
        ]),
    )

    assert sorted([
        Record(fielddate=b'2020-04-27', puid_tree=b'\tstaff name 1 staff_login_1\t', _acl=_FULL_ACCESS_GROUP),
        Record(fielddate=b'2020-04-27', puid_tree=b'\tstaff name 2 (staff_login_2)\t', _acl=_FULL_ACCESS_GROUP + b',@staff_login_2'),
    ]) == sorted(result)


def test_should_aggregate_tasks_by_date():
    result = nile_ut.yt_run(
        report._aggregate_tasks,
        log=nile_ut.Table([
            Record(fielddate=b'2020-04-27', puid_tree=b'1', task_name_tree=b'1', region_name_tree=b'1', schedule_tree=b'1',
                   holiday=True, basic_unit_to_rub=1, quantity=1.0, time_spent_sec=1, cost_rub=1.0, cost_basic_unit=11, _acl=b'acl 1'),
            Record(fielddate=b'2020-04-27', puid_tree=b'1', task_name_tree=b'1', region_name_tree=b'1', schedule_tree=b'1',
                   holiday=True, basic_unit_to_rub=1, quantity=2.0, time_spent_sec=2, cost_rub=2.0, cost_basic_unit=22, _acl=b'acl 1'),
            Record(fielddate=b'2020-04-28', puid_tree=b'1', task_name_tree=b'1', region_name_tree=b'1', schedule_tree=b'1',
                   holiday=True, basic_unit_to_rub=1, quantity=4.0, time_spent_sec=4, cost_rub=4.0, cost_basic_unit=44, _acl=b'acl 1'),
        ])
    )

    assert sorted([
        Record(fielddate=b'2020-04-27', puid_tree=b'1', task_name_tree=b'1', region_name_tree=b'1', schedule_tree=b'1',
               holiday=True, basic_unit_to_rub=1, quantity_total=3.0, time_spent_total_sec=3, cost_total_rub=3.0, cost_total_basic_unit=33, _acl=b'acl 1'),
        Record(fielddate=b'2020-04-28', puid_tree=b'1', task_name_tree=b'1', region_name_tree=b'1', schedule_tree=b'1',
               holiday=True, basic_unit_to_rub=1, quantity_total=4.0, time_spent_total_sec=4, cost_total_rub=4.0, cost_total_basic_unit=44, _acl=b'acl 1'),
    ]) == sorted(result)


def test_should_aggregate_tasks_by_puid_tree():
    result = nile_ut.yt_run(
        report._aggregate_tasks,
        log=nile_ut.Table([
            Record(fielddate=b'2020-04-27', puid_tree=b'1', task_name_tree=b'1', region_name_tree=b'1', schedule_tree=b'1',
                   holiday=True, basic_unit_to_rub=1, quantity=1.0, time_spent_sec=1, cost_rub=1.0, cost_basic_unit=11, _acl=b'acl 1'),
            Record(fielddate=b'2020-04-27', puid_tree=b'2', task_name_tree=b'1', region_name_tree=b'1', schedule_tree=b'1',
                   holiday=True, basic_unit_to_rub=1, quantity=2.0, time_spent_sec=2, cost_rub=2.0, cost_basic_unit=22, _acl=b'acl 1'),
            Record(fielddate=b'2020-04-27', puid_tree=b'1', task_name_tree=b'1', region_name_tree=b'1', schedule_tree=b'1',
                   holiday=True, basic_unit_to_rub=1, quantity=4.0, time_spent_sec=4, cost_rub=4.0, cost_basic_unit=44, _acl=b'acl 1'),
        ])
    )

    assert sorted([
        Record(fielddate=b'2020-04-27', puid_tree=b'1', task_name_tree=b'1', region_name_tree=b'1', schedule_tree=b'1',
               holiday=True, basic_unit_to_rub=1, quantity_total=5.0, time_spent_total_sec=5, cost_total_rub=5.0, cost_total_basic_unit=55, _acl=b'acl 1'),
        Record(fielddate=b'2020-04-27', puid_tree=b'2', task_name_tree=b'1', region_name_tree=b'1', schedule_tree=b'1',
               holiday=True, basic_unit_to_rub=1, quantity_total=2.0, time_spent_total_sec=2, cost_total_rub=2.0, cost_total_basic_unit=22, _acl=b'acl 1'),
    ]) == sorted(result)


def test_should_aggregate_tasks_by_task_name():
    result = nile_ut.yt_run(
        report._aggregate_tasks,
        log=nile_ut.Table([
            Record(fielddate=b'2020-04-27', puid_tree=b'1', task_name_tree=b'1', region_name_tree=b'1', schedule_tree=b'1',
                   holiday=True, basic_unit_to_rub=1, quantity=1.0, time_spent_sec=1, cost_rub=1.0, cost_basic_unit=11, _acl=b'acl 1'),
            Record(fielddate=b'2020-04-27', puid_tree=b'1', task_name_tree=b'2', region_name_tree=b'1', schedule_tree=b'1',
                   holiday=True, basic_unit_to_rub=1, quantity=2.0, time_spent_sec=2, cost_rub=2.0, cost_basic_unit=22, _acl=b'acl 1'),
            Record(fielddate=b'2020-04-27', puid_tree=b'1', task_name_tree=b'2', region_name_tree=b'1', schedule_tree=b'1',
                   holiday=True, basic_unit_to_rub=1, quantity=4.0, time_spent_sec=4, cost_rub=4.0, cost_basic_unit=44, _acl=b'acl 1'),
        ])
    )

    assert sorted([
        Record(fielddate=b'2020-04-27', puid_tree=b'1', task_name_tree=b'1', region_name_tree=b'1', schedule_tree=b'1',
               holiday=True, basic_unit_to_rub=1, quantity_total=1.0, time_spent_total_sec=1, cost_total_rub=1.0, cost_total_basic_unit=11, _acl=b'acl 1'),
        Record(fielddate=b'2020-04-27', puid_tree=b'1', task_name_tree=b'2', region_name_tree=b'1', schedule_tree=b'1',
               holiday=True, basic_unit_to_rub=1, quantity_total=6.0, time_spent_total_sec=6, cost_total_rub=6.0, cost_total_basic_unit=66, _acl=b'acl 1'),
    ]) == sorted(result)


def test_should_aggregate_tasks_by_region():
    result = nile_ut.yt_run(
        report._aggregate_tasks,
        log=nile_ut.Table([
            Record(fielddate=b'2020-04-27', puid_tree=b'1', task_name_tree=b'1', region_name_tree=b'1', schedule_tree=b'1',
                   holiday=True, basic_unit_to_rub=1, quantity=1.0, time_spent_sec=1, cost_rub=1.0, cost_basic_unit=11, _acl=b'acl 1'),
            Record(fielddate=b'2020-04-27', puid_tree=b'1', task_name_tree=b'1', region_name_tree=b'2', schedule_tree=b'1',
                   holiday=True, basic_unit_to_rub=1, quantity=2.0, time_spent_sec=2, cost_rub=2.0, cost_basic_unit=22, _acl=b'acl 1'),
            Record(fielddate=b'2020-04-27', puid_tree=b'1', task_name_tree=b'1', region_name_tree=b'1', schedule_tree=b'1',
                   holiday=True, basic_unit_to_rub=1, quantity=4.0, time_spent_sec=4, cost_rub=4.0, cost_basic_unit=44, _acl=b'acl 1'),
        ])
    )

    assert sorted([
        Record(fielddate=b'2020-04-27', puid_tree=b'1', task_name_tree=b'1', region_name_tree=b'1', schedule_tree=b'1',
               holiday=True, basic_unit_to_rub=1, quantity_total=5.0, time_spent_total_sec=5, cost_total_rub=5.0, cost_total_basic_unit=55, _acl=b'acl 1'),
        Record(fielddate=b'2020-04-27', puid_tree=b'1', task_name_tree=b'1', region_name_tree=b'2', schedule_tree=b'1',
               holiday=True, basic_unit_to_rub=1, quantity_total=2.0, time_spent_total_sec=2, cost_total_rub=2.0, cost_total_basic_unit=22, _acl=b'acl 1'),
    ]) == sorted(result)


def test_should_aggregate_tasks_by_schedule():
    result = nile_ut.yt_run(
        report._aggregate_tasks,
        log=nile_ut.Table([
            Record(fielddate=b'2020-04-27', puid_tree=b'1', task_name_tree=b'1', region_name_tree=b'1', schedule_tree=b'2',
                   holiday=True, basic_unit_to_rub=1, quantity=1.0, time_spent_sec=1, cost_rub=1.0, cost_basic_unit=11, _acl=b'acl 1'),
            Record(fielddate=b'2020-04-27', puid_tree=b'1', task_name_tree=b'1', region_name_tree=b'1', schedule_tree=b'1',
                   holiday=True, basic_unit_to_rub=1, quantity=2.0, time_spent_sec=2, cost_rub=2.0, cost_basic_unit=22, _acl=b'acl 1'),
            Record(fielddate=b'2020-04-27', puid_tree=b'1', task_name_tree=b'1', region_name_tree=b'1', schedule_tree=b'1',
                   holiday=True, basic_unit_to_rub=1, quantity=4.0, time_spent_sec=4, cost_rub=4.0, cost_basic_unit=44, _acl=b'acl 1'),
        ])
    )

    assert sorted([
        Record(fielddate=b'2020-04-27', puid_tree=b'1', task_name_tree=b'1', region_name_tree=b'1', schedule_tree=b'1',
               holiday=True, basic_unit_to_rub=1, quantity_total=6.0, time_spent_total_sec=6, cost_total_rub=6.0, cost_total_basic_unit=66, _acl=b'acl 1'),
        Record(fielddate=b'2020-04-27', puid_tree=b'1', task_name_tree=b'1', region_name_tree=b'1', schedule_tree=b'2',
               holiday=True, basic_unit_to_rub=1, quantity_total=1.0, time_spent_total_sec=1, cost_total_rub=1.0, cost_total_basic_unit=11, _acl=b'acl 1'),
    ]) == sorted(result)


def test_should_aggregate_tasks_by_holiday():
    result = nile_ut.yt_run(
        report._aggregate_tasks,
        log=nile_ut.Table([
            Record(fielddate=b'2020-04-27', puid_tree=b'1', task_name_tree=b'1', region_name_tree=b'1', schedule_tree=b'1',
                   holiday=True, basic_unit_to_rub=1, quantity=1.0, time_spent_sec=1, cost_rub=1.0, cost_basic_unit=11, _acl=b'acl 1'),
            Record(fielddate=b'2020-04-27', puid_tree=b'1', task_name_tree=b'1', region_name_tree=b'1', schedule_tree=b'1',
                   holiday=None, basic_unit_to_rub=1, quantity=2.0, time_spent_sec=2, cost_rub=2.0, cost_basic_unit=22, _acl=b'acl 1'),
            Record(fielddate=b'2020-04-27', puid_tree=b'1', task_name_tree=b'1', region_name_tree=b'1', schedule_tree=b'1',
                   holiday=True, basic_unit_to_rub=1, quantity=4.0, time_spent_sec=4, cost_rub=4.0, cost_basic_unit=44, _acl=b'acl 1'),
        ])
    )

    assert sorted([
        Record(fielddate=b'2020-04-27', puid_tree=b'1', task_name_tree=b'1', region_name_tree=b'1', schedule_tree=b'1',
               holiday=True, basic_unit_to_rub=1, quantity_total=5.0, time_spent_total_sec=5, cost_total_rub=5.0, cost_total_basic_unit=55, _acl=b'acl 1'),
        Record(fielddate=b'2020-04-27', puid_tree=b'1', task_name_tree=b'1', region_name_tree=b'1', schedule_tree=b'1',
               holiday=None, basic_unit_to_rub=1, quantity_total=2.0, time_spent_total_sec=2, cost_total_rub=2.0, cost_total_basic_unit=22, _acl=b'acl 1'),
    ]) == sorted(result)


def test_should_aggregate_tasks_by_acl():
    result = nile_ut.yt_run(
        report._aggregate_tasks,
        log=nile_ut.Table([
            Record(fielddate=b'2020-04-27', puid_tree=b'1', task_name_tree=b'1', region_name_tree=b'1', schedule_tree=b'1',
                   holiday=True, basic_unit_to_rub=1, quantity=1.0, time_spent_sec=1, cost_rub=1.0, cost_basic_unit=11, _acl=b'acl 1'),
            Record(fielddate=b'2020-04-27', puid_tree=b'1', task_name_tree=b'1', region_name_tree=b'1', schedule_tree=b'1',
                   holiday=True, basic_unit_to_rub=1, quantity=2.0, time_spent_sec=2, cost_rub=2.0, cost_basic_unit=22, _acl=b'acl 1'),
            Record(fielddate=b'2020-04-27', puid_tree=b'1', task_name_tree=b'1', region_name_tree=b'1', schedule_tree=b'1',
                   holiday=True, basic_unit_to_rub=1, quantity=4.0, time_spent_sec=4, cost_rub=4.0, cost_basic_unit=44, _acl=b'acl 2'),
        ])
    )

    assert sorted([
        Record(fielddate=b'2020-04-27', puid_tree=b'1', task_name_tree=b'1', region_name_tree=b'1', schedule_tree=b'1',
               holiday=True, basic_unit_to_rub=1, quantity_total=3.0, time_spent_total_sec=3, cost_total_rub=3.0, cost_total_basic_unit=33, _acl=b'acl 1'),
        Record(fielddate=b'2020-04-27', puid_tree=b'1', task_name_tree=b'1', region_name_tree=b'1', schedule_tree=b'1',
               holiday=True, basic_unit_to_rub=1, quantity_total=4.0, time_spent_total_sec=4, cost_total_rub=4.0, cost_total_basic_unit=44, _acl=b'acl 2'),
    ]) == sorted(result)


def test_should_prepare_report(geobase_fixture):
    'Checks that the whole job works as it is expected. (an integration test)'

    result = nile_ut.yt_run(
        report.prepare_report,
        nile_ut.Job(),
        '2020-04-27',
        nile_ut.Table([
            Record(fielddate=b'2020-04-27', puid=1, task_id=b'task', mrc_region=b'mrc region', quantity=1.0, lat_min=1.2, lon_min=1.4, lat_max=1.6, lon_max=1.8),
            # Unknown region
            Record(fielddate=b'2020-04-27', puid=1, task_id=b'task', mrc_region=b'mrc region', quantity=2.0, lat_min=2, lon_min=2, lat_max=2, lon_max=2),
        ]),
        major_regions_map=nile_ut.Table([
            # fixture returns round(lat + lon) = round((lat_min + lat_max) / 2 + (lon_min + lon_max) / 2)
            Record(region_id=b'3', region_tree=b'\t10000\t'),
            Record(region_id=b'3', region_tree=b'\t10000\tregion\t'),
        ]),
        task_tariff_map=nile_ut.Table([
            Record(task_id=b'task', task_name_tree=b'\t__MRC_REGION__\t',       seconds_per_task=1),
            Record(task_id=b'task', task_name_tree=b'\t__MRC_REGION__\ttask\t', seconds_per_task=1),
            Record(task_id=_TASK_ID_CURRENT_EMPLOYEE, task_name_tree=b'current employee', seconds_per_task=0),
        ]),
        puid_map=nile_ut.Table([
            Record(puid=1, puid_tree=b'\tall\t'),
            Record(puid=1, puid_tree=b'\tall\tstaff name (staff_login)\t'),
        ]),
        linked_accounts=nile_ut.Table([
            Record(puid=1, login=b'nmaps login', staff_uid=101, staff_login=b'staff_login', is_primary_link=True),
        ]),
        work_schedule=nile_ut.Table([
            Record(staff_login=b'staff_login', schedule=b'schedule'),
        ]),
        user_holidays=nile_ut.Table([
            Record(staff_login=b'staff_login', date=b'2020-04-27'),
        ]),
        basic_units=nile_ut.Table([
            Record(staff_login=b'staff_login', basic_unit_to_rub=0.1),
        ]),
        staff_dump=nile_ut.Table([
            Record(login=b'staff_login', quit_at=None),
        ]),
    )

    assert sorted([
        # Contains information about both entries from the log (with and without region)
        Record(fielddate=b'2020-04-27', region_name_tree=_DEFAULT_REGION_TREE, task_name_tree=b'\tmrc region\t', puid_tree=b'\tall\t', holiday=None,
               basic_unit_to_rub=None, schedule_tree=b'\tall\t', quantity_total=3.0, time_spent_total_sec=3.0, _acl=b'full_access_group',
               cost_total_rub=3.0 * TARIFF_RUB_PER_SEC * _HOLIDAY_PAY_RATE_FACTOR,
               cost_total_basic_unit=30 * TARIFF_RUB_PER_SEC),
        Record(fielddate=b'2020-04-27', region_name_tree=_DEFAULT_REGION_TREE, task_name_tree=b'\tmrc region\t', puid_tree=b'\tall\t', holiday=None,
               basic_unit_to_rub=None, schedule_tree=b'\tall\tschedule\t', quantity_total=3.0, time_spent_total_sec=3.0, _acl=b'full_access_group',
               cost_total_rub=3.0 * TARIFF_RUB_PER_SEC * _HOLIDAY_PAY_RATE_FACTOR,
               cost_total_basic_unit=30 * TARIFF_RUB_PER_SEC),
        Record(fielddate=b'2020-04-27', region_name_tree=_DEFAULT_REGION_TREE, task_name_tree=b'\tmrc region\t', puid_tree=b'\tall\tstaff name (staff_login)\t', holiday=True,
               basic_unit_to_rub=0.1, schedule_tree=b'\tall\t', quantity_total=3.0, time_spent_total_sec=3.0, _acl=b'full_access_group,@staff_login',
               cost_total_rub=3.0 * TARIFF_RUB_PER_SEC * _HOLIDAY_PAY_RATE_FACTOR,
               cost_total_basic_unit=30 * TARIFF_RUB_PER_SEC),
        Record(fielddate=b'2020-04-27', region_name_tree=_DEFAULT_REGION_TREE, task_name_tree=b'\tmrc region\t', puid_tree=b'\tall\tstaff name (staff_login)\t', holiday=True,
               basic_unit_to_rub=0.1, schedule_tree=b'\tall\tschedule\t', quantity_total=3.0, time_spent_total_sec=3.0, _acl=b'full_access_group,@staff_login',
               cost_total_rub=3.0 * TARIFF_RUB_PER_SEC * _HOLIDAY_PAY_RATE_FACTOR,
               cost_total_basic_unit=30 * TARIFF_RUB_PER_SEC),
        Record(fielddate=b'2020-04-27', region_name_tree=_DEFAULT_REGION_TREE, task_name_tree=b'\tmrc region\ttask\t', puid_tree=b'\tall\t', holiday=None,
               basic_unit_to_rub=None, schedule_tree=b'\tall\t', quantity_total=3.0, time_spent_total_sec=3.0, _acl=b'full_access_group',
               cost_total_rub=3.0 * TARIFF_RUB_PER_SEC * _HOLIDAY_PAY_RATE_FACTOR,
               cost_total_basic_unit=30 * TARIFF_RUB_PER_SEC),
        Record(fielddate=b'2020-04-27', region_name_tree=_DEFAULT_REGION_TREE, task_name_tree=b'\tmrc region\ttask\t', puid_tree=b'\tall\t', holiday=None,
               basic_unit_to_rub=None, schedule_tree=b'\tall\tschedule\t', quantity_total=3.0, time_spent_total_sec=3.0, _acl=b'full_access_group',
               cost_total_rub=3.0 * TARIFF_RUB_PER_SEC * _HOLIDAY_PAY_RATE_FACTOR,
               cost_total_basic_unit=30 * TARIFF_RUB_PER_SEC),
        Record(fielddate=b'2020-04-27', region_name_tree=_DEFAULT_REGION_TREE, task_name_tree=b'\tmrc region\ttask\t', puid_tree=b'\tall\tstaff name (staff_login)\t', holiday=True,
               basic_unit_to_rub=0.1, schedule_tree=b'\tall\t', quantity_total=3.0, time_spent_total_sec=3.0, _acl=b'full_access_group,@staff_login',
               cost_total_rub=3.0 * TARIFF_RUB_PER_SEC * _HOLIDAY_PAY_RATE_FACTOR,
               cost_total_basic_unit=30 * TARIFF_RUB_PER_SEC),
        Record(fielddate=b'2020-04-27', region_name_tree=_DEFAULT_REGION_TREE, task_name_tree=b'\tmrc region\ttask\t', puid_tree=b'\tall\tstaff name (staff_login)\t', holiday=True,
               basic_unit_to_rub=0.1, schedule_tree=b'\tall\tschedule\t', quantity_total=3.0, time_spent_total_sec=3.0, _acl=b'full_access_group,@staff_login',
               cost_total_rub=3.0 * TARIFF_RUB_PER_SEC * _HOLIDAY_PAY_RATE_FACTOR,
               cost_total_basic_unit=30 * TARIFF_RUB_PER_SEC),

        # Contains information about the first entry from the log (with region)
        Record(fielddate=b'2020-04-27', region_name_tree=_DEFAULT_REGION_TREE + b'region\t', task_name_tree=b'\tmrc region\t', puid_tree=b'\tall\t', holiday=None,
               basic_unit_to_rub=None, schedule_tree=b'\tall\t', quantity_total=1.0, time_spent_total_sec=1.0, _acl=b'full_access_group',
               cost_total_rub=1.0 * TARIFF_RUB_PER_SEC * _HOLIDAY_PAY_RATE_FACTOR,
               cost_total_basic_unit=10 * TARIFF_RUB_PER_SEC),
        Record(fielddate=b'2020-04-27', region_name_tree=_DEFAULT_REGION_TREE + b'region\t', task_name_tree=b'\tmrc region\t', puid_tree=b'\tall\t', holiday=None,
               basic_unit_to_rub=None, schedule_tree=b'\tall\tschedule\t', quantity_total=1.0, time_spent_total_sec=1.0, _acl=b'full_access_group',
               cost_total_rub=1.0 * TARIFF_RUB_PER_SEC * _HOLIDAY_PAY_RATE_FACTOR,
               cost_total_basic_unit=10 * TARIFF_RUB_PER_SEC),
        Record(fielddate=b'2020-04-27', region_name_tree=_DEFAULT_REGION_TREE + b'region\t', task_name_tree=b'\tmrc region\t', puid_tree=b'\tall\tstaff name (staff_login)\t', holiday=True,
               basic_unit_to_rub=0.1, schedule_tree=b'\tall\t', quantity_total=1.0, time_spent_total_sec=1.0, _acl=b'full_access_group,@staff_login',
               cost_total_rub=1.0 * TARIFF_RUB_PER_SEC * _HOLIDAY_PAY_RATE_FACTOR,
               cost_total_basic_unit=10 * TARIFF_RUB_PER_SEC),
        Record(fielddate=b'2020-04-27', region_name_tree=_DEFAULT_REGION_TREE + b'region\t', task_name_tree=b'\tmrc region\t', puid_tree=b'\tall\tstaff name (staff_login)\t', holiday=True,
               basic_unit_to_rub=0.1, schedule_tree=b'\tall\tschedule\t', quantity_total=1.0, time_spent_total_sec=1.0, _acl=b'full_access_group,@staff_login',
               cost_total_rub=1.0 * TARIFF_RUB_PER_SEC * _HOLIDAY_PAY_RATE_FACTOR,
               cost_total_basic_unit=10 * TARIFF_RUB_PER_SEC),
        Record(fielddate=b'2020-04-27', region_name_tree=_DEFAULT_REGION_TREE + b'region\t', task_name_tree=b'\tmrc region\ttask\t', puid_tree=b'\tall\t', holiday=None,
               basic_unit_to_rub=None, schedule_tree=b'\tall\t', quantity_total=1.0, time_spent_total_sec=1.0, _acl=b'full_access_group',
               cost_total_rub=1.0 * TARIFF_RUB_PER_SEC * _HOLIDAY_PAY_RATE_FACTOR,
               cost_total_basic_unit=10 * TARIFF_RUB_PER_SEC),
        Record(fielddate=b'2020-04-27', region_name_tree=_DEFAULT_REGION_TREE + b'region\t', task_name_tree=b'\tmrc region\ttask\t', puid_tree=b'\tall\t', holiday=None,
               basic_unit_to_rub=None, schedule_tree=b'\tall\tschedule\t', quantity_total=1.0, time_spent_total_sec=1.0, _acl=b'full_access_group',
               cost_total_rub=1.0 * TARIFF_RUB_PER_SEC * _HOLIDAY_PAY_RATE_FACTOR,
               cost_total_basic_unit=10 * TARIFF_RUB_PER_SEC),
        Record(fielddate=b'2020-04-27', region_name_tree=_DEFAULT_REGION_TREE + b'region\t', task_name_tree=b'\tmrc region\ttask\t', puid_tree=b'\tall\tstaff name (staff_login)\t', holiday=True,
               basic_unit_to_rub=0.1, schedule_tree=b'\tall\t', quantity_total=1.0, time_spent_total_sec=1.0, _acl=b'full_access_group,@staff_login',
               cost_total_rub=1.0 * TARIFF_RUB_PER_SEC * _HOLIDAY_PAY_RATE_FACTOR,
               cost_total_basic_unit=10 * TARIFF_RUB_PER_SEC),
        Record(fielddate=b'2020-04-27', region_name_tree=_DEFAULT_REGION_TREE + b'region\t', task_name_tree=b'\tmrc region\ttask\t', puid_tree=b'\tall\tstaff name (staff_login)\t', holiday=True,
               basic_unit_to_rub=0.1, schedule_tree=b'\tall\tschedule\t', quantity_total=1.0, time_spent_total_sec=1.0, _acl=b'full_access_group,@staff_login',
               cost_total_rub=1.0 * TARIFF_RUB_PER_SEC * _HOLIDAY_PAY_RATE_FACTOR,
               cost_total_basic_unit=10 * TARIFF_RUB_PER_SEC),

        # Contains information about current employees
        Record(fielddate=b'2020-04-27', region_name_tree=_DEFAULT_REGION_TREE, task_name_tree=b'current employee', puid_tree=b'\tall\t', holiday=None,
               basic_unit_to_rub=None, schedule_tree=b'\tall\t', quantity_total=0.0, time_spent_total_sec=0, _acl=b'full_access_group',
               cost_total_basic_unit=0.0, cost_total_rub=0.0),
        Record(fielddate=b'2020-04-27', region_name_tree=_DEFAULT_REGION_TREE, task_name_tree=b'current employee', puid_tree=b'\tall\t', holiday=None,
               basic_unit_to_rub=None, schedule_tree=b'\tall\tschedule\t', quantity_total=0.0, time_spent_total_sec=0, _acl=b'full_access_group',
               cost_total_basic_unit=0.0, cost_total_rub=0.0),
        Record(fielddate=b'2020-04-27', region_name_tree=_DEFAULT_REGION_TREE, task_name_tree=b'current employee', puid_tree=b'\tall\tstaff name (staff_login)\t', holiday=True,
               basic_unit_to_rub=0.1, schedule_tree=b'\tall\t', quantity_total=0.0, time_spent_total_sec=0, _acl=b'full_access_group,@staff_login',
               cost_total_basic_unit=0.0, cost_total_rub=0.0),
        Record(fielddate=b'2020-04-27', region_name_tree=_DEFAULT_REGION_TREE, task_name_tree=b'current employee', puid_tree=b'\tall\tstaff name (staff_login)\t', holiday=True,
               basic_unit_to_rub=0.1, schedule_tree=b'\tall\tschedule\t', quantity_total=0.0, time_spent_total_sec=0, _acl=b'full_access_group,@staff_login',
               cost_total_basic_unit=0.0, cost_total_rub=0.0),
    ]) == sorted(result)
