from maps.wikimap.stat.tasks_payment.dictionaries.libs.users import normalize_yalogin
from maps.wikimap.stat.tasks_payment.tasks_logging.libs.geometry import append_bbox, geo_wkb_geometry_to_geo_bbox
from maps.wikimap.stat.tasks_payment.tasks_logging.libs.parsing import isotime_normalizer

from nile.api.v1 import extractors, filters
from cyson import UInt


_TASK_ID = b'autocart/bld'


def _filter_by_date(column_name, date):
    return filters.custom(
        lambda value: value is not None and value.startswith(date),
        column_name
    )


def _extract_as_uint(column_name):
    return extractors.custom(
        lambda value: UInt(value),
        column_name
    )


def _get_tasks_dumped_at(date, assessors_table):
    '''
    assessors_table:
    | dump_date | login | shape | ... |
    |-----------+-------+-------+-----|
    | ...       | ...   | ...   | ... |

    result:
    | dump_date | login | shape |
    |-----------+-------+-------|
    | date-...  | ...   | ...   |
    '''
    return assessors_table.filter(
        _filter_by_date('dump_date', date)
    ).project(
        'dump_date', 'shape',
        login=extractors.custom(normalize_yalogin, 'login')
    )


def _check_for_absent_puids(records):
    '''
    Checks that each record in `records` has a non empty 'puid' field.
    '''
    for record in records:
        assert 'puid' in record and record['puid'] is not None, f"Login {record['login']} has no puid."
        yield record


def _login_to_puid(log, nmaps_users_dump_log):
    '''
    log:
    | dump_date | login | shape |
    |-----------+-------+-------|
    | date-...  | ...   | ...   |

    nmaps_users_dump_log:
    | um_login | puid | ... |
    |----------+------+-----|
    | ...      | ...  | ... |

    result:
    | dump_date | puid | shape |
    |-----------+------+-------|
    | ...       | ...  | ...   |
    '''
    nmaps_users_dump_log = nmaps_users_dump_log.project(
        login=extractors.custom(normalize_yalogin, 'um_login'),
        puid='puid'
    )

    return log.join(
        nmaps_users_dump_log,
        by='login',
        type='left',
        assume_unique_right=True
    ).map(
        _check_for_absent_puids
    ).project(
        'dump_date', 'puid', 'shape'
    )


def _convert_to_result_format(log):
    '''
    log:
    | dump_date | puid (string) | shape | lat_min | lon_min | lat_max | lon_max |
    |-----------+---------------+-------+---------+---------+---------+---------|
    | ...       | ...           | ...   | ...     | ...     | ...     | ...     |

    Result:
    | iso_datetime | puid (uint) | task_id  | quantity | geom  | lat_min | lon_min | lat_max | lon_max |
    |--------------+-------------+----------+----------+-------+---------+---------+---------+---------|
    | dump_date    | ...         | _TASK_ID |        1 | shape | ...     | ...     | ...     | ...     |
    '''
    return log.project(
        iso_datetime=extractors.custom(isotime_normalizer(), 'dump_date'),
        puid=_extract_as_uint('puid'),
        task_id=extractors.const(_TASK_ID),
        quantity=extractors.const(1.0),
        geom='shape',
        lat_min='lat_min',
        lon_min='lon_min',
        lat_max='lat_max',
        lon_max='lon_max'
    )


def make_log(date, assessors_table, nmaps_users_dump_log):
    date = date.encode('utf-8')

    log = _get_tasks_dumped_at(date, assessors_table)
    log = _login_to_puid(log, nmaps_users_dump_log)
    log = append_bbox(log, 'shape', geo_wkb_geometry_to_geo_bbox)
    return _convert_to_result_format(log)
