#include <maps/wikimap/ugc/backoffice/src/lib/globals.h>
#include <maps/wikimap/ugc/backoffice/src/lib/params.h>
#include <maps/wikimap/ugc/backoffice/src/lib/assignments/load.h>
#include <maps/wikimap/ugc/backoffice/src/lib/assignments/modify.h>
#include <maps/wikimap/ugc/backoffice/src/lib/contributions/modify.h>
#include <maps/wikimap/ugc/backoffice/src/lib/links.h>

#include <maps/wikimap/ugc/libs/common/config.h>
#include <maps/wikimap/ugc/libs/common/dbqueries.h>
#include <maps/wikimap/ugc/libs/common/locks.h>
#include <maps/wikimap/ugc/libs/common/options.h>
#include <yandex/maps/proto/ugc_account/backoffice.pb.h>

#include <maps/infra/yacare/include/limit_rate.h>
#include <maps/infra/yacare/include/params/tvm.h>
#include <maps/infra/yacare/include/tvm.h>
#include <maps/infra/yacare/include/yacare.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/locale/include/convert.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/libs/pgpool3utils/include/yandex/maps/pgpool3utils/dynamic_pool_holder.h>
#include <maps/libs/pgpool3utils/include/yandex/maps/pgpool3utils/pg_advisory_mutex.h>


using namespace maps::wiki::ugc;
using namespace maps::wiki::ugc::backoffice;

namespace {

const size_t THREADS = 64;
const size_t BACKLOG = 1024;
yacare::ThreadPool MAIN_THREAD_POOL("mainThreadPool", THREADS, BACKLOG);

const std::string& CONFIG_PATH = "/etc/yandex/maps/ugc_backoffice/ugc_backoffice.conf";

std::unique_ptr<Globals> g_globals;

Globals& globals()
{
    REQUIRE(g_globals, "Globals are not initialized");
    return *g_globals;
}

} // namespace

using namespace yandex::maps::proto::ugc_account::backoffice;

// The number of seconds before the shutdown to serve requests while not
// responding to /ping.
YCR_OPTIONS.shutdown().grace_period() = 10;


yacare::VirtualHost yacareVhost {
    yacare::VirtualHost::SLB { "core-ugc-backoffice" }
};

YCR_SET_DEFAULT(
    yacareVhost,
    yacare::Tvm2ServiceCheck("maps-core-ugc-backoffice"),
    yacare::LimitRate().resource("maps_core_ugc_backoffice")
);

YCR_USE(MAIN_THREAD_POOL) {

YCR_RESPOND_TO("PUT /v1/assignments/create", task_id, tvmId, ttl=4838400)
try {
    Task task;
    if (!task.ParseFromString(TString{request.body()})) {
        throw yacare::errors::BadRequest() << "Bad request body: " << request.body();
    }
    createAssignments(
        globals().pools,
        globals().dryRun(),
        task_id,
        task,
        tvmId,
        std::chrono::seconds(ttl),
        globals().assignmentsValidator);

    response.setStatus(maps::http::Status::Created);
} catch (const maps::locale::LocaleParsingError& e) {
    throw yacare::errors::BadRequest() << e.what();
}

YCR_RESPOND_TO("PUT /v1/assignments/done", uid, task_id, tvmId) {
    globals().assignmentsValidator.checkId(tvmId, task_id.value());
    auto txn = globals().pools.at(uid).masterWriteableTransaction();
    doneAssignment(*txn, uid, task_id);
    finishTxn(*txn, globals().dryRun());
}

YCR_RESPOND_TO("DELETE /v1/assignments/delete", uid, task_id, tvmId) {
    globals().assignmentsValidator.checkId(tvmId, task_id.value());
    auto txn = globals().pools.at(uid).masterWriteableTransaction();
    deleteAssignment(*txn, uid, task_id);
    finishTxn(*txn, globals().dryRun());
}

YCR_RESPOND_TO(
    "GET /v1/assignments/status", uid, task_id, tvmId,
    YCR_LIMIT_RATE(resource("maps_core_ugc_backoffice_status"))
) {
    globals().assignmentsValidator.checkId(tvmId, task_id.value());
    auto txn = globals().pools.at(uid).slaveTransaction();
    response << getAssignmentStatus(*txn, uid, task_id);
}

YCR_RESPOND_TO("PUT /v1/contributions/modify", uid, tvmId) try {
    ModifyContributions modifyContributions;
    if (!modifyContributions.ParseFromString(TString{request.body()})) {
        throw yacare::errors::BadRequest() << "Bad request body: " << request.body();
    }
    DEBUG() << "Request body: " << modifyContributions.DebugString();
    if (modifyContributions.modify_contribution_size() == 0) {
        DEBUG() << "Empty data in body for uid " << uid;
        return;
    }

    auto txn = globals().pools.at(uid).masterWriteableTransaction();
    // Using a lock in case something fails and we get multiple requests
    // corresponding to one uid at a time.
    maps::pgp3utils::PgAdvisoryXactMutex uid_lock(
        globals().pools.at(uid),
        locks::lockIdFromUid(
            locks::CONTRIBUTION_MODIFICATION_DBMUTEXGROUPID, uid));
    if (!uid_lock.try_lock()) {
        throw yacare::errors::Conflict()
            << "Contribution modification for uid=" << uid
            << " is already in progress";
    }
    updateDb(*txn, modifyContributions, uid, tvmId, globals().contributionsValidator);
    finishTxn(*txn, globals().dryRun());

    response.setStatus(maps::http::Status::Created);

} catch (const maps::locale::LocaleParsingError& e) {
    throw yacare::errors::BadRequest() << e.what();
}

YCR_RESPOND_TO("PUT /v1/link/add", uid, task_id, contribution_id, tvmId) {
    globals().assignmentsValidator.checkId(tvmId, task_id.value());
    globals().contributionsValidator.checkId(tvmId, contribution_id.value());
    auto txn = globals().pools.at(uid).masterWriteableTransaction();
    addLink(*txn, uid, task_id, contribution_id);
    finishTxn(*txn, globals().dryRun());

    response.setStatus(maps::http::Status::Created);
}

YCR_RESPOND_TO("DELETE /v1/link/delete", uid, task_id, contribution_id, tvmId) {
    globals().assignmentsValidator.checkId(tvmId, task_id.value());
    globals().contributionsValidator.checkId(tvmId, contribution_id.value());
    auto txn = globals().pools.at(uid).masterWriteableTransaction();
    deleteLink(*txn, uid, task_id, contribution_id);
    finishTxn(*txn, globals().dryRun());
}

} // YCR_USE MAIN_THREAD_POOL

YCR_MAIN(argc, argv) {
    try {
        auto options = parseOptions(argc, argv, CONFIG_PATH);
        g_globals = Globals::create(options.confPath, options.dryRun);

        yacare::run();

        return EXIT_SUCCESS;
    } catch (const maps::Exception& e) {
        ERROR() << "maps::Exception: " << e;
    } catch (const std::exception& e) {
        ERROR() << "std::exception: " << e.what();
    }

    return EXIT_FAILURE;
}
