from aiohttp import web

from maps_adv.adv_store.v2.lib.db import DB
from maps_adv.adv_store.v2.lib.domains import (
    CampaignsDomain,
    EventsDomain,
    ModerationDomain,
)

from . import campaigns, events, middlewares, moderation, ping
from .api_providers import (
    CampaignsApiProvider,
    EventsApiProvider,
    ModerationApiProvider,
)

__all__ = ["create"]


def create(
    db: DB,
    campaigns_domain: CampaignsDomain,
    moderation_domain: ModerationDomain,
    events_domain: EventsDomain,
    ping_v1_url: str,
) -> web.Application:
    ping_resources = ping.Resources(db, ping_v1_url=ping_v1_url)
    campaigns_resources = campaigns.Resources(CampaignsApiProvider(campaigns_domain))
    moderation_resources = moderation.Resources(
        ModerationApiProvider(moderation_domain)
    )
    events_resources = events.Resources(EventsApiProvider(events_domain))

    api = web.Application(
        middlewares=[
            middlewares.handle_validation_exception,
            middlewares.handle_object_not_found,
            middlewares.handle_domain_exception,
        ]
    )
    api.add_routes(
        [
            web.get(r"/ping", ping_resources.ping),
            web.get(
                r"/campaigns/{campaign_id:\d+}/", campaigns_resources.retrieve_campaign
            ),
            web.put(
                r"/campaigns/{campaign_id:\d+}/", campaigns_resources.update_campaign
            ),
            web.post(r"/campaigns/", campaigns_resources.create_campaign),
            web.put(
                r"/campaigns/{campaign_id:\d+}/status/", campaigns_resources.set_status
            ),
            web.put(
                r"/campaigns/{campaign_id:\d+}/paid-till/",
                campaigns_resources.set_paid_till,
            ),
            web.post(
                r"/campaigns/list-short/", campaigns_resources.list_short_campaings
            ),
            web.post(r"/campaigns/list/", campaigns_resources.list_campaigns_by_orders),
            web.get(
                r"/campaigns/charger/cpm/",
                campaigns_resources.list_campaigns_for_charger,
            ),
            web.get(
                r"/campaigns/charger/cpa/",
                campaigns_resources.list_campaigns_for_charger_cpa,
            ),
            web.get(
                r"/campaigns/charger/fix/",
                campaigns_resources.list_campaigns_for_charger_fix,
            ),
            web.put(r"/campaigns/stop/", campaigns_resources.stop_campaigns),
            web.post(
                r"/campaigns/predict-daily-budget/",
                campaigns_resources.predict_daily_budget,
            ),
            web.get(
                r"/campaigns/export/", campaigns_resources.list_campaigns_for_export
            ),
            web.get(
                r"/campaigns/budget-analysis/",
                campaigns_resources.list_campaigns_for_budget_analysis,
            ),
            web.get(r"/used-audience/", campaigns_resources.list_used_audiences),
            web.post(
                r"/orders/campaigns-summary/",
                campaigns_resources.list_campaigns_summary,
            ),
            web.get(r"/moderation/list/", moderation_resources.list_campaigns),
            web.post(r"/moderation/review/", moderation_resources.review_campaign),
            web.post(
                r"/events/by-order-ids/",
                events_resources.retrieve_campaigns_events_by_orders,
            ),
            web.post(
                r"/create-campaign-events/", events_resources.create_campaign_events
            ),
            web.post(
                r"/campaigns/monitoring-info/",
                campaigns_resources.retrieve_campaign_data_for_monitorings,
            ),
        ]
    )

    return api
