import json
from datetime import datetime

from aiohttp import web
from aiohttp.web import Request, Response, json_response

from maps_adv.adv_store.v2.lib.api.api_providers.campaigns import CampaignsApiProvider


class Resources:
    _provider: CampaignsApiProvider

    def __init__(self, provider: CampaignsApiProvider):
        self._provider = provider

    async def create_campaign(self, request: Request) -> Response:
        body = await request.read()
        result = await self._provider.create_campaign(body)
        return web.Response(body=result, status=201)

    async def retrieve_campaign(self, request: Request) -> Response:
        campaign_id = int(request.match_info["campaign_id"])
        result = await self._provider.retrieve_campaign(campaign_id=campaign_id)
        return web.Response(body=result, status=200)

    async def update_campaign(self, request: Request) -> Response:
        campaign_id = int(request.match_info["campaign_id"])
        body = await request.read()
        result = await self._provider.update_campaign(body, campaign_id=campaign_id)
        return web.Response(body=result, status=200)

    async def set_status(self, request: Request) -> Response:
        campaign_id = int(request.match_info["campaign_id"])
        body = await request.read()
        await self._provider.set_status(body, campaign_id=campaign_id)
        return web.Response(status=200)

    async def set_paid_till(self, request: Request) -> Response:
        campaign_id = int(request.match_info["campaign_id"])
        body = await request.read()
        await self._provider.set_paid_till(body, campaign_id=campaign_id)
        return web.Response(status=200)

    async def list_short_campaings(self, request: Request) -> Response:
        body = await request.read()
        result = await self._provider.list_short_campaigns(body)
        return web.Response(body=result, status=200)

    async def list_campaigns_by_orders(self, request: Request) -> Response:
        body = await request.read()
        result = await self._provider.list_campaigns_by_orders(body)
        return web.Response(body=result, status=200)

    async def list_campaigns_summary(self, request: Request) -> Response:
        body = await request.read()
        result = await self._provider.list_campaigns_summary(body)
        return web.Response(body=result, status=200)

    async def list_campaigns_for_charger(self, request: Request) -> Response:
        try:
            active_at_ts = int(request.query["active_at"])
        except (KeyError, ValueError):
            return web.Response(status=400)

        active_at = datetime.fromtimestamp(active_at_ts)
        result = await self._provider.list_campaigns_for_charger(active_at=active_at)
        return web.Response(body=result, status=200)

    async def list_campaigns_for_charger_cpa(self, request: Request) -> Response:
        try:
            active_at_ts = int(request.query["active_at"])
        except (KeyError, ValueError):
            return web.Response(status=400)

        active_at = datetime.fromtimestamp(active_at_ts)
        result = await self._provider.list_campaigns_for_charger_cpa(
            active_at=active_at
        )
        return web.Response(body=result, status=200)

    async def list_campaigns_for_charger_fix(self, request: Request) -> Response:
        try:
            active_at_ts = int(request.query["active_at"])
        except (KeyError, ValueError):
            return web.Response(status=400)

        active_at = datetime.fromtimestamp(active_at_ts)
        result = await self._provider.list_campaigns_for_charger_fix(
            active_at=active_at
        )
        return web.Response(body=result, status=200)

    async def stop_campaigns(self, request: Request) -> Response:
        body = await request.read()
        await self._provider.stop_campaigns(body)
        return web.Response(status=200)

    async def list_campaigns_for_export(self, request: Request) -> Response:
        result = await self._provider.list_campaigns_for_export()
        return web.Response(body=result, status=200)

    async def predict_daily_budget(self, request: Request) -> Response:
        body = await request.read()
        result = await self._provider.predict_daily_budget(body)
        return Response(body=result, status=200)

    async def list_used_audiences(self, request: Request) -> json:
        result = await self._provider.list_used_audiences()
        return json_response(data=result, status=200)

    async def list_campaigns_for_budget_analysis(self, request: Request) -> Response:
        result = await self._provider.list_campaigns_for_budget_analysis()
        return Response(body=result, status=200)

    async def retrieve_campaign_data_for_monitorings(
        self, request: Request
    ) -> Response:
        body = await request.read()
        result = await self._provider.retrieve_campaign_data_for_monitorings(body)
        return Response(body=result, status=200)
