client_or_agency_exists = """
    SELECT EXISTS (
        SELECT 1
        FROM clients
        WHERE id = $1 AND is_agency = $2
    )
"""

find_client_by_id = """
    SELECT
        id,
        name,
        email,
        phone,
        is_agency,
        account_manager_id,
        domain,
        partner_agency_id,
        COALESCE(has_accepted_offer, FALSE) as has_accepted_offer,
        representatives
    FROM clients
    WHERE id = $1
"""

insert_client = """
    INSERT INTO clients (
        id, name, email, phone, account_manager_id, domain, partner_agency_id, is_agency, has_accepted_offer, created_from_cabinet
    )
    VALUES ($1, $2, $3, $4, $5, $6, $7, FALSE, $8, $9)
    RETURNING id, name, email, phone, account_manager_id, partner_agency_id, is_agency, has_accepted_offer, created_from_cabinet
"""


def upsert_client(keys, **kwargs):
    return f"""
        INSERT INTO clients ({",".join(keys)})
        VALUES ({",".join(f"${i + 1}" for i in range(len(kwargs)))})
        ON CONFLICT (id) DO UPDATE
        SET {",".join(f"{key} = excluded.{key}" for key in keys)}
        RETURNING *
    """


set_account_manager_for_client = """
    UPDATE clients
    SET
        account_manager_id = $2
    WHERE id = $1
    RETURNING *
"""

set_representatives_for_client = """
    UPDATE clients
    SET
        representatives = $2
    WHERE id = $1
    RETURNING *
"""

list_client_ids = """
    SELECT array_agg(id)
    FROM clients
"""

list_agencies = """
    SELECT id, name, email, phone, COALESCE(has_accepted_offer, FALSE) as has_accepted_offer
    FROM clients
    WHERE is_agency IS TRUE
    ORDER BY id
"""

list_agency_clients = """
    SELECT
        clients.id,
        clients.name,
        clients.email,
        clients.phone,
        COUNT(orders.id) AS orders_count,
        clients.account_manager_id,
        clients.partner_agency_id,
        COALESCE(clients.has_accepted_offer, FALSE) as has_accepted_offer
    FROM clients
    LEFT JOIN agency_clients
        ON clients.id = agency_clients.client_id
    LEFT JOIN orders
        ON clients.id = orders.client_id
            AND (
                $1::integer IS NULL AND orders.agency_id IS NULL
                OR orders.agency_id = $1
            )
    WHERE (
        $1::integer IS NULL
        AND
        agency_clients.agency_id IS NULL
        AND
        agency_clients.client_id IS NOT NULL
        OR
        agency_clients.agency_id = $1
    )
    GROUP BY clients.id, clients.name, clients.email, clients.phone
    ORDER BY clients.id
"""

list_account_manager_clients = """
    SELECT
        clients.id,
        clients.name,
        clients.email,
        clients.phone,
        clients.account_manager_id,
        clients.partner_agency_id,
        COALESCE(clients.has_accepted_offer, FALSE) as has_accepted_offer
    FROM clients
    WHERE clients.account_manager_id = $1
    ORDER BY clients.id
"""

upsert_agency_client = """
    INSERT INTO agency_clients (agency_id, client_id)
    VALUES ($1, $2)
    ON CONFLICT DO NOTHING
"""

remove_clients_from_agency = """
    DELETE FROM agency_clients
    WHERE client_id = ANY($1::bigint[]) AND agency_id = $2
"""

remove_clients_from_internal = """
    DELETE FROM agency_clients
    WHERE client_id = ANY($1::bigint[]) AND agency_id IS NULL
"""

client_is_in_agency = """
    SELECT EXISTS (
        SELECT 1
        FROM agency_clients
        WHERE client_id = $1 AND agency_id = $2
    )
"""

client_is_in_internal = """
    SELECT EXISTS (
        SELECT 1
        FROM agency_clients
        WHERE client_id = $1 AND agency_id IS NULL
    )
"""

list_clients_with_agencies = """
    SELECT DISTINCT client_id
    FROM agency_clients
    WHERE client_id = ANY($1::bigint[])
"""

list_clients_with_orders_with_agency = """
    SELECT DISTINCT client_id
    FROM orders
    WHERE client_id = ANY($1::bigint[])
    AND (($2::integer IS NULL and agency_id IS NULL) OR agency_id = $2::integer)
"""

list_existing_client_ids = """
    SELECT array_agg(id)
    FROM clients
    WHERE id = ANY($1::bigint[])
"""

set_client_has_accepted_offer = """
    UPDATE clients SET has_accepted_offer = TRUE
    WHERE id = $1 and is_agency = $2
    RETURNING *
"""

list_clients = """
    SELECT
        clients.id,
        clients.name,
        clients.email,
        clients.phone,
        NULL as orders_count,
        clients.account_manager_id,
        clients.partner_agency_id,
        COALESCE(clients.has_accepted_offer, FALSE) as has_accepted_offer
    FROM clients
    WHERE is_agency IS FALSE
    ORDER BY clients.id
"""
